<?php
/**
 * @author          jtotal <support@jtotal.org>
 * @link            https://jtotal.org
 * @copyright       Copyright © 2020 JTOTAL All Rights Reserved
 * @license         GNU GPLv3 <http://www.gnu.org/licenses/gpl.html> or later
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Installer\Installer;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Installer\InstallerAdapter;
use Joomla\CMS\Installer\InstallerScriptInterface;
use Joomla\Database\DatabaseInterface;

return new class implements InstallerScriptInterface {
    protected $minimumPhp = '8.3.0';
    protected $minimumJoomla = '5.4.0';
    
    public $name = 'JT| FA parser';
    public $element = 'jtfaparser';
    public $alias = 'plg_content_jtfaparser';
    public $extension_type = 'plugin';
    public $plugin_folder = 'content';
    public $extname = '';
    public $module_position = 'status';
    public $client_id = 1;
    public $install_type = 'install';
    public $show_message = true;
    public $autopublish = true;
    protected DatabaseInterface $db;
    public $app = null;
    public $installedVersion;

    public function __construct()
    {
        $this->extname = $this->extname ?: $this->alias;
        $this->db = Factory::getContainer()->get(DatabaseInterface::class);
        $this->app = Factory::getApplication();
        $this->installedVersion = $this->getVersion($this->getInstalledXMLFile());
        $lang = $this->app->getLanguage();
        $lang->load('plg_system_jtframework', JPATH_BASE);
    }

    public function jtfwInstalled(): bool
    {
        return is_file(JPATH_PLUGINS . '/system/jtframework/autoload.php');
    }


    public function install(InstallerAdapter $adapter): bool
    {
        return true;
    }
    public function update(InstallerAdapter $adapter): bool
    {
        return true;
    }
    public function uninstall(InstallerAdapter $adapter): bool
    {
        return true;
    }


    public function onBeforeInstall(): bool
    {
        if (!$this->jtfwInstalled()) {
            $this->app->enqueueMessage(
                Text::_('JT framework required. Please, <a target="_blank" href="https://users.jtotal.org/SOFT/framework/JTframework/pkg_jtfw_current.zip">download at jtotal.org</a> and install it before.'),
                'error'
            );
            return false;
        } else {
            $this->checkOldFrameworkUpdate('https://users.jotal.org/UPDATE/COMPONENTS/JTFA/update-jtfa.xml');
        }
        return true;
    }

    public function checkOldFrameworkUpdate($siteToDelete = false): void
    {
        if (!$siteToDelete) {
            return;
        }

        $query = $this->db->getQuery(true);
        $query->select('COUNT(*) AS count')
        ->from($this->db->quoteName('#__update_sites'))
        ->where($this->db->quoteName('location') . ' = ' . $this->db->quote($siteToDelete));

        $this->db->setQuery($query);
        $result = $this->db->loadResult();

        if ($result > 0) {
            $query = $this->db->getQuery(true);
            $query->delete($this->db->quoteName('#__update_sites'))
                ->where($this->db->quoteName('location') . ' = ' . $this->db->quote($siteToDelete));
            $this->db->setQuery($query);
            $this->db->execute();
            $this->app->enqueueMessage(Text::_('Social2s OLD Update Sites deleted'), 'info');
        }
    }

    public function preflight(string $type, InstallerAdapter $adapter): bool
    {
        if (!in_array($type, ['install', 'update'])) {
            return true;
        }

        $this->app->getLanguage()->load('plg_system_jtframework', JPATH_PLUGINS . '/system/jtframework/language');

        if ($this->show_message && $this->isInstalled()) {
            $this->install_type = 'update';
        }

        return $this->onBeforeInstall();
    }

    public function postflight(string $type, InstallerAdapter $adapter): bool
    {
        if (!in_array($type, ['install', 'update'])) {
            return true;
        }

        if ($this->onAfterInstall() === false) {
            return false;
        }

        if ($type === 'install' && $this->autopublish) {
            $this->publishExtension();
        }

        if ($this->show_message) {
            $this->addInstalledMessage();
        }

        $cacheFactory = Factory::getContainer()->get(\Joomla\CMS\Cache\CacheControllerFactoryInterface::class);
        $cacheController = $cacheFactory->createCacheController('callback');
        $cacheController->clean('com_plugins');
        $cacheController->clean('_system');
        
        return true;
    }

    public function publishExtension()
    {
        $this->publishPlugin();
    }

    public function publishPlugin()
    {
        $query = $this->db->getQuery(true)
            ->update($this->db->quoteName('#__extensions'))
            ->set($this->db->quoteName('enabled') . ' = 1')
            ->where($this->db->quoteName('type') . ' = ' . $this->db->quote('plugin'))
            ->where($this->db->quoteName('element') . ' = ' . $this->db->quote($this->element))
            ->where($this->db->quoteName('folder') . ' = ' . $this->db->quote($this->plugin_folder));
        $this->db->setQuery($query);
        $this->db->execute();
    }

    public function addInstalledMessage()
    {
        $this->app->enqueueMessage(
            '<strong>JTOTAL ' . Text::_($this->extname) . '</strong> ' .
            '<strong>' . $this->getVersion() . '</strong> '
        );

        $this->app->enqueueMessage(
            Text::_($this->install_type == 'update' ? 'JT_EXTENSION_UPDATED' : 'JT_EXTENSION_INSTALLED')
        );
    }

    public function getVersion($file = '')
    {
        $file = $file ?: $this->getCurrentXMLFile();

        if (!is_file($file)) {
            return '';
        }

        $xml = Installer::parseXMLInstallFile($file);

        if (!$xml || !isset($xml['version'])) {
            return '';
        }

        return $xml['version'];
    }

    public function getCurrentXMLFile()
    {
        return $this->getXMLFile(__DIR__);
    }

    public function getXMLFile($folder)
    {
        return $folder . '/' . $this->extname . '.xml';
    }

    public function isInstalled(): bool
    {
        if (!is_file($this->getInstalledXMLFile())) {
            return false;
        }

        $query = $this->db->getQuery(true)
        ->select('extension_id')
        ->from($this->db->quoteName('#__extensions'))
        ->where($this->db->quoteName('type') . ' = ' . $this->db->quote($this->extension_type))
        ->where($this->db->quoteName('element') . ' = ' . $this->db->quote($this->getElementName()));
        $this->db->setQuery($query, 0, 1);
        $result = $this->db->loadResult();

        return !empty($result);
    }

    protected function onAfterInstall(): bool
    {
        return true;
    }

    public function getInstalledXMLFile()
    {
        return $this->getXMLFile($this->getMainFolder());
    }

    public function getMainFolder()
    {
        switch ($this->extension_type) {
            case 'plugin':
                return JPATH_SITE . '/plugins/' . $this->plugin_folder . '/' . $this->extname;

            case 'component':
                return JPATH_ADMINISTRATOR . '/components/com_' . $this->extname;

            case 'module':
                return JPATH_ADMINISTRATOR . '/modules/mod_' . $this->extname;

            case 'library':
                return JPATH_SITE . '/libraries/' . $this->extname;
        }
        return '';
    }

    public function getElementName($type = null, $extname = null)
    {
        $type = is_null($type) ? $this->extension_type : $type;
        $extname = is_null($extname) ? $this->extname : $extname;

        switch ($type) {
            case 'component':
                return 'com_' . $extname;

            case 'module':
                return 'mod_' . $extname;

            case 'plugin':
            default:
                return $extname;
        }
    }

};
