<?php
/**
 * @author          jtotal <support@jtotal.org>
 * @link            https://jtotal.org
 * @copyright       Copyright © 2020 JTOTAL All Rights Reserved
 * @license         GNU GPLv3 <http://www.gnu.org/licenses/gpl.html> or later
 */

namespace Jtotal\Plugin\EditorsXtd\Jtfabutton\Script;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Installer\Installer;
use Joomla\CMS\Installer\InstallerAdapter;
use Joomla\CMS\Installer\InstallerScriptInterface;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Application\CMSApplication;
use Joomla\Database\DatabaseDriver;
use Joomla\DI\Container;
use Joomla\DI\ServiceProviderInterface;

return new class implements InstallerScriptInterface, ServiceProviderInterface {
    public $name = 'JT | FA button';
    public $element = 'jtfabutton';
    public $alias = 'JT | FA button';
    public $extname = '';
    public $extension_type = 'plugin';
    public $plugin_folder = 'editors-xtd';
    public $module_position = 'status';
    public $client_id = 1;
    public $install_type = 'install';
    public $show_message = true;
    public $autopublish = true;
    private DatabaseDriver $db;
    private CMSApplication $app;
    public $installedVersion;

    public function __construct()
    {
        $this->extname = $this->extname ?: $this->element;
        $this->db = Factory::getContainer()->get(DatabaseDriver::class);
        $this->app = Factory::getApplication();
        $this->installedVersion = $this->getVersion($this->getInstalledXMLFile());
        $lang = $this->app->getLanguage();
        $lang->load('plg_system_jtframework', JPATH_BASE);
    }

    public function register(Container $container): void
    {
        // Required for ServiceProviderInterface
    }

    public function jtfwInstalled()
    {
        if (is_file(JPATH_PLUGINS . '/system/jtframework/autoload.php')) {
            return true;
        }

        return false;
    }

    public function install(InstallerAdapter $adapter): bool
    {
        return true;
    }

    public function update(InstallerAdapter $adapter): bool
    {
        return true;
    }

    public function uninstall(InstallerAdapter $adapter): bool
    {
        return true;
    }

    public function preflight(string $type, InstallerAdapter $adapter): bool
    {
        if (!in_array($type, array('install', 'update'))) {
            return true;
        }

        $this->app->getLanguage()->load('plg_system_jtframework', JPATH_PLUGINS . '/system/jtframework/language');

        if ($this->show_message && $this->isInstalled()) {
            $this->install_type = 'update';
        }

        if (!$this->jtfwInstalled()) {
            $this->app->enqueueMessage(Text::_('JT framework required. Please, <a target="_blank" href="https://users.jtotal.org/SOFT/framework/JTframework/pkg_jtfw_current.zip">download at jtotal.org</a> and install it before.'), 'error');
            return false;
        }

        return true;
    }

    public function postflight(string $type, InstallerAdapter $adapter): bool
    {
        if (!in_array($type, array('install', 'update'))) {
            return true;
        }

        if ($type == 'install' && $this->autopublish) {
            $this->publishExtension();
        }

        if ($this->show_message) {
            $this->addInstalledMessage();
        }

        Factory::getCache()->clean('com_plugins');
        Factory::getCache()->clean('_system');

        return true;
    }

    public function isNewer()
    {
        if (!$installed_version = $this->getVersion($this->getInstalledXMLFile())) {
            return true;
        }

        $package_version = $this->getVersion();

        return version_compare($installed_version, $package_version, '<=');
    }

    public function getVersion($file = '')
    {
        $file = $file ?: $this->getCurrentXMLFile();

        if (!is_file($file)) {
            return '';
        }

        $xml = Installer::parseXMLInstallFile($file);

        if (!$xml || !isset($xml['version'])) {
            return '';
        }

        return $xml['version'];
    }

    public function isInstalled()
    {
        if (!is_file($this->getInstalledXMLFile())) {
            return false;
        }

        $query = $this->db->getQuery(true)
            ->select('extension_id')
            ->from($this->db->quoteName('#__extensions'))
            ->where($this->db->quoteName('type') . ' = ' . $this->db->quote($this->extension_type))
            ->where($this->db->quoteName('element') . ' = ' . $this->db->quote($this->getElementName()));
        
        $result = $this->db->setQuery($query, 0, 1)->loadResult();

        return !empty($result);
    }

    public function getMainFolder()
    {
        switch ($this->extension_type) {
            case 'plugin':
                return JPATH_SITE . '/plugins/' . $this->plugin_folder . '/' . $this->extname;

            case 'component':
                return JPATH_ADMINISTRATOR . '/components/com_' . $this->extname;

            case 'module':
                return JPATH_ADMINISTRATOR . '/modules/mod_' . $this->extname;

            case 'library':
                return JPATH_SITE . '/libraries/' . $this->extname;
        }
    }

    public function getInstalledXMLFile()
    {
        return $this->getXMLFile($this->getMainFolder());
    }

    public function getCurrentXMLFile()
    {
        return $this->getXMLFile(__DIR__);
    }

    public function getXMLFile($folder)
    {
        return $folder . '/' . $this->extname . '.xml';
    }

    public function publishExtension()
    {
        switch ($this->extension_type) {
            case 'plugin':
                $this->publishPlugin();
                break;

            case 'module':
                $this->publishModule();
                break;
        }
    }

    public function publishPlugin()
    {
        $query = $this->db->getQuery(true)
            ->update($this->db->quoteName('#__extensions'))
            ->set($this->db->quoteName('enabled') . ' = 1')
            ->where($this->db->quoteName('type') . ' = ' . $this->db->quote('plugin'))
            ->where($this->db->quoteName('element') . ' = ' . $this->db->quote($this->element))
            ->where($this->db->quoteName('folder') . ' = ' . $this->db->quote($this->plugin_folder));
        $this->db->setQuery($query);
        $this->db->execute();
    }

    public function publishModule()
    {
        $query = $this->db->getQuery(true)
            ->select('id')
            ->from($this->db->quoteName('#__modules'))
            ->where($this->db->quoteName('module') . ' = ' . $this->db->quote('mod_' . $this->extname))
            ->where($this->db->quoteName('client_id') . ' = ' . (int) $this->client_id);
        
        $id = $this->db->setQuery($query, 0, 1)->loadResult();

        if (!$id) {
            return;
        }

        $query->clear()
            ->select('moduleid')
            ->from($this->db->quoteName('#__modules_menu'))
            ->where($this->db->quoteName('moduleid') . ' = ' . (int) $id);
        
        $exists = $this->db->setQuery($query, 0, 1)->loadResult();

        if ($exists) {
            return;
        }

        $query->clear()
            ->select('ordering')
            ->from($this->db->quoteName('#__modules'))
            ->where($this->db->quoteName('position') . ' = ' . $this->db->quote($this->module_position))
            ->where($this->db->quoteName('client_id') . ' = ' . (int) $this->client_id)
            ->order('ordering DESC');
        
        $ordering = $this->db->setQuery($query, 0, 1)->loadResult();
        $ordering++;

        $query->clear()
            ->update($this->db->quoteName('#__modules'))
            ->set($this->db->quoteName('published') . ' = 1')
            ->set($this->db->quoteName('ordering') . ' = ' . (int) $ordering)
            ->set($this->db->quoteName('position') . ' = ' . $this->db->quote($this->module_position))
            ->where($this->db->quoteName('id') . ' = ' . (int) $id);
        $this->db->setQuery($query);
        $this->db->execute();

        $query->clear()
            ->insert($this->db->quoteName('#__modules_menu'))
            ->columns(array($this->db->quoteName('moduleid'), $this->db->quoteName('menuid')))
            ->values((int) $id . ', 0');
        $this->db->setQuery($query);
        $this->db->execute();
    }

    public function addInstalledMessage()
    {
        $this->app->enqueueMessage(
            '<strong>JTOTAL ' . Text::_($this->extname) . '</strong> ' .
            '<strong>' . $this->getVersion() . '</strong> '
        );

        $this->app->enqueueMessage(
            Text::_($this->install_type == 'update' ? 'JT_EXTENSION_UPDATED' : 'JT_EXTENSION_INSTALLED')
        );
    }

    public function getElementName($type = null, $extname = null)
    {
        $type = is_null($type) ? $this->extension_type : $type;
        $extname = is_null($extname) ? $this->extname : $extname;

        switch ($type) {
            case 'component':
                return 'com_' . $extname;

            case 'module':
                return 'mod_' . $extname;

            case 'plugin':
            default:
                return $extname;
        }
    }
};
