<?php
/**
 * @package     Jtotal.Library
 * @subpackage  jtlibs
 * @copyright   Copyright (C) 2025 Jtotal. All rights reserved.
 * @license     GNU General Public License version 2 or later; see https://jtotal.org/LICENSE.txt
 */

namespace Jtotal\Library\Jtlibs\Script;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Installer\InstallerAdapter;
use Joomla\CMS\Installer\InstallerScriptInterface;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;

return new class implements InstallerScriptInterface {

    public bool $autopublish = false;
    public string $name = 'JT Library';
    public string $alias = 'jtlibs';
    public string $extension_type = 'library';
    public string $plugin_folder = '';
    public bool $soft_break = true;
    
    public function install(InstallerAdapter $adapter): bool
    {
        //Log::add('Library jtlibs installed correctly.', Log::INFO, 'jerror');
        return true;
    }

    public function update(InstallerAdapter $adapter): bool
    {
        //Log::add('Library jtlibs updated correctly.', Log::INFO, 'jerror');
        return true;
    }

    public function uninstall(InstallerAdapter $adapter): bool
    {
        //Log::add('Library jtlibs uninstalled correctly.', Log::INFO, 'jerror');
        return true;
    }

    public function preflight(string $type, InstallerAdapter $adapter): bool
    {
        // Verificar si es una versión más nueva antes de instalar/actualizar
        if ($type === 'install' || $type === 'update') {
            if (!$this->isNewer($adapter)) {
                $app = Factory::getApplication();
                $app->enqueueMessage(Text::_('JT_SYSTEM_UPDATED_FAILED_NEW'), 'info');
                return false;
            }
        }

        return true;
    }

    public function postflight(string $type, InstallerAdapter $adapter): bool
    {
        $app = Factory::getApplication();

        if ($type === 'install') {
            $app->enqueueMessage(
                Text::sprintf('<i class="fa-solid fa-book"></i> %s has been installed successfully!', $this->name),
                'success'
            );
        } elseif ($type === 'update') {
            $app->enqueueMessage(
                Text::sprintf('<i class="fa-solid fa-book"></i> %s has been updated successfully!', $this->name),
                'info'
            );
        }

        return true;
    }

    private function isNewer(InstallerAdapter $adapter): bool
    {
        // Obtener la versión del manifest actual
        $manifest = $adapter->getManifest();
        $newVersion = (string) $manifest->version;

        // Obtener la versión instalada actualmente
        $db = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true)
            ->select($db->quoteName('manifest_cache'))
            ->from($db->quoteName('#__extensions'))
            ->where($db->quoteName('type') . ' = ' . $db->quote('library'))
            ->where($db->quoteName('element') . ' = ' . $db->quote($this->alias));

        $db->setQuery($query);
        $manifestCache = $db->loadResult();

        if (!$manifestCache) {
            // Si no hay versión instalada, permitir instalación
            return true;
        }

        $cacheData = json_decode($manifestCache, true);
        $currentVersion = $cacheData['version'] ?? '0.0.0';

        // Comparar versiones
        return version_compare($newVersion, $currentVersion, '>=');
    }

    private function parseXMLInstallFile(string $file): array|false
    {
        return \Joomla\CMS\Installer\Installer::parseXMLInstallFile($file);
    }
};
