<?php

/**
 * @author          jtotal <support@jtotal.org>
 * @link            https://jtotal.org
 * @copyright       Copyright © 2020 JTOTAL All Rights Reserved
 * @license         GNU GPLv3 <http://www.gnu.org/licenses/gpl.html> or later
 */

namespace jtotal\Jtfw;

use Joomla\CMS\Filesystem\Folder;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Response\JsonResponse;
use Joomla\CMS\Session\Session;

defined('_JEXEC') or die('Restricted access');

class PlgSystemJTFramework extends CMSPlugin
{
    /**
     *  Auto load plugin language
     *
     *  @var  boolean
     */
    protected $autoloadLanguage = true;

    /**
     *  The Joomla Application object
     *
     *  @var  object
     */
    protected $app;

    /**
     *  Plugin constructor
     *
     *  @param  mixed   &$subject
     *  @param  array   $config
     */
    public function __construct(&$subject, $config = array())
    {
        // Declare extension logger
        Log::addLogger(
            array('text_file' => 'plg_system_jtframework.php'),
            Log::ALL,
            array('jtframework')
        );

        // execute parent constructor
        parent::__construct($subject, $config);
    }

    public function onAfterInitialise()
    {
        //
    }

    public function onAjaxPluginParams()
    {
        Session::checkToken('request') or jexit(Text::_('JINVALID_TOKEN'));
        return json_encode("nook");
    }

    public function onAjaxJTFramework()
    {
        try {
            // Verificar token CSRF
            if (!Session::checkToken('request')) {
                throw new \Exception(Text::_('JINVALID_TOKEN'), 403);
            }

            // Verificar que el usuario tenga permisos
            if (!$this->app->getIdentity()->authorise('core.admin')) {
                throw new \Exception('No autorizado', 403);
            }

            // Obtener tarea
            $task = $this->app->input->get('task', null);
            
            if (empty($task)) {
                throw new \Exception('Tarea no especificada', 400);
            }

            // Verificar si existe el método
            $taskMethod = 'ajaxTask' . $task;
            
            if (!method_exists($this, $taskMethod)) {
                throw new \Exception('Tarea no encontrada: ' . $task, 404);
            }

            // Ejecutar la tarea
            $result = $this->$taskMethod();
            
            // Registrar en el log
            Log::add(
                sprintf('Tarea %s ejecutada correctamente', $task),
                Log::INFO,
                'jtframework'
            );

            // Devolver respuesta exitosa
            return new JsonResponse($result, null, false, true);
        } catch (\Exception $e) {
            // Registrar el error
            Log::add(
                sprintf('Error en AJAX JTFramework: %s', $e->getMessage()),
                Log::ERROR,
                'jtframework'
            );

            // Devolver respuesta de error
            return new JsonResponse(null, $e->getMessage(), true, false);
        }
    }

    /**
     * Clears cache
     *
     * @return void
     */
    private function ajaxTaskClearCache(): bool
    {
        $directory = $this->app->input->get('directory', null);

        if (empty($directory)) {
            throw new \Exception('Directorio no especificado', 400);
        }

        // Validar el directorio (seguridad)
        $directory = basename($directory); // Evitar directory traversal
        
        if (!preg_match('/^[a-zA-Z0-9_-]+$/', $directory)) {
            throw new \Exception('Nombre de directorio inválido', 400);
        }

        $cachePath = JPATH_SITE . '/cache/' . $directory;

        if (!Folder::exists($cachePath)) {
            // Si el directorio no existe, consideramos que ya está "limpio"
            return true;
        }

        $result = Folder::delete($cachePath);
        
        if (!$result) {
            throw new \Exception('No se pudo eliminar el directorio de cache', 500);
        }

        return true;
    }
}
