<?php
/**
 * @package     Jtotal.Plugin
 * @subpackage  System.jtframework
 * @copyright   Copyright (C) 2025 Jtotal. All rights reserved.
 * @license     GNU General Public License version 2 or later; see [https://jtotal.org/LICENSE.txt](https://jtotal.org/LICENSE.txt)
 */

namespace Jtotal\Plugin\System\JTFramework\Script;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Installer\Installer;
use Joomla\CMS\Installer\InstallerAdapter;
use Joomla\CMS\Installer\InstallerScriptInterface;
use Joomla\CMS\Language\Text;

return new class implements InstallerScriptInterface {

    public bool $autopublish = true;
    public string $name = 'JT | FrameWork';
    public string $alias = 'jtframework';
    public string $extension_type = 'plugin';
    public string $plugin_folder = 'system';
    //public string $minJtfwVersion = '5.0.0';
    
    public function install(InstallerAdapter $adapter): bool
    {
        return true;
    }

    public function update(InstallerAdapter $adapter): bool
    {

        return true;
    }

    public function uninstall(InstallerAdapter $adapter): bool
    {
        return true;
    }

    public function preflight(string $type, InstallerAdapter $adapter): bool
    {
        $app = Factory::getApplication();

        return true;
    }

    public function postflight(string $type, InstallerAdapter $adapter): bool
    {
        if ($type === 'install' && $this->autopublish) {
            $this->publishPlugin();
            $link = 'index.php?option=com_plugins&view=plugins&filter[search]=Jtframework';
            $message = Text::sprintf('FrameWork installed. <a class="btn btn-success" href="%s">Configure PRO license here</a>', $link);
            Factory::getApplication()->enqueueMessage($message, 'info');
        }


        $this->checkOldFrameworkUpdate('https://users.jtotal.org/SOFT/UPDATES/FRAMEWORK/JTframework/jtframework-updatev4.xml');
        $this->checkOldFrameworkUpdate('https://users.jtotal.org/SOFT/UPDATES/FRAMEWORK/JTframework/jtframework-updatev5.xml');

        $this->checkFrameworkUpdate(
            'https://users.jtotal.org/SOFT/UPDATES/FRAMEWORK/JTframework/jtframework-update.xml'
        );

        return true;
    }

    private function checkOldFrameworkUpdate(string $siteToDelete): void
    {
        if (!$siteToDelete) {
            return;
        }

        $db = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true)
            ->select('COUNT(*) AS count')
            ->from('#__update_sites')
            ->where($db->quoteName('location') . ' = ' . $db->quote($siteToDelete));

        $db->setQuery($query);
        $result = $db->loadResult();

        if ($result > 0) {
            $query = $db->getQuery(true)
                ->delete($db->quoteName('#__update_sites'))
                ->where($db->quoteName('location') . ' = ' . $db->quote($siteToDelete));
            $db->setQuery($query);
            $db->execute();
            Factory::getApplication()->enqueueMessage(Text::_($siteToDelete . ' OLD Update Sites deleted :)'), 'info');
        }
    }

    private function checkFrameworkUpdate(string $siteToFindAndUpdate): void
    {
        $db = Factory::getDbo();
        
        // Check if site exists
        $query = $db->getQuery(true)
            ->select('COUNT(*) AS count')
            ->from('#__update_sites')
            ->where($db->quoteName('location') . ' = ' . $db->quote($siteToFindAndUpdate));

        $db->setQuery($query);
        $siteExists = $db->loadResult();

        if ($siteExists == 0) {
            // Get extension_id for this plugin
            $query = $db->getQuery(true)
                ->select('extension_id')
                ->from('#__extensions')
                ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'))
                ->where($db->quoteName('element') . ' = ' . $db->quote($this->alias))
                ->where($db->quoteName('folder') . ' = ' . $db->quote($this->plugin_folder));
            $db->setQuery($query);
            $extensionId = $db->loadResult();

            if ($extensionId) {
                // Insert new update site
                $query = $db->getQuery(true)
                    ->insert('#__update_sites')
                    ->columns([$db->quoteName('name'), $db->quoteName('type'), $db->quoteName('location'), $db->quoteName('enabled')])
                    ->values($db->quote($this->name) . ', ' . $db->quote('extension') . ', ' . $db->quote($siteToFindAndUpdate) . ', 1');
                $db->setQuery($query);
                $db->execute();
                
                $updateSiteId = $db->insertid();
                
                // Link update site to extension
                $query = $db->getQuery(true)
                    ->insert('#__update_sites_extensions')
                    ->columns([$db->quoteName('update_site_id'), $db->quoteName('extension_id')])
                    ->values($updateSiteId . ', ' . $extensionId);
                $db->setQuery($query);
                $db->execute();
                
                Factory::getApplication()->enqueueMessage(Text::_($this->name . ' Update Site created :)'), 'info');
            }
        }
    }

    
    private function publishPlugin(): void
    {
        $db = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true)
            ->update('#__extensions')
            ->set($db->quoteName('enabled') . ' = 1')
            ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'))
            ->where($db->quoteName('element') . ' = ' . $db->quote($this->alias))
            ->where($db->quoteName('folder') . ' = ' . $db->quote($this->plugin_folder));
        $db->setQuery($query);
        $db->execute();
    }


    public function isNewer(): bool
    {
        if (!$installed_version = $this->getVersion($this->getInstalledXMLFile())) {
            return true;
        }

        $package_version = $this->getVersion();

        return version_compare($installed_version, $package_version, '<=');
    }

    public function getVersion(string $file = ''): string
    {
        $file = $file ?: $this->getCurrentXMLFile();

        if (!is_file($file)) {
            return '';
        }

        $xml = Installer::parseXMLInstallFile($file);

        if (!$xml || !isset($xml['version'])) {
            return '';
        }

        return $xml['version'];
    }
    
    public function getInstalledXMLFile(): string
    {
        return $this->getXMLFile($this->getMainFolder());
    }

    public function getCurrentXMLFile(): string
    {
        return $this->getXMLFile(__DIR__);
    }

    public function getXMLFile(string $folder): string
    {
        return $folder . '/' . $this->alias . '.xml';
    }

    public function getMainFolder(): string
    {
        return JPATH_SITE . '/plugins/' . $this->plugin_folder . '/' . $this->alias;
    }


};
