<?php
/**
 * @package     Jtotal.Plugin
 * @subpackage  Content.jtattach
 * @copyright   Copyright (C) 2025 Jtotal. All rights reserved.
 * @license     GNU General Public License version 2 or later; see [https://jtotal.org/LICENSE.txt](https://jtotal.org/LICENSE.txt)
 */

namespace Jtotal\Plugin\System\Social2sog\Script;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Installer\InstallerAdapter;
use Joomla\CMS\Installer\InstallerScriptInterface;
use Joomla\CMS\Language\Text;

return new class implements InstallerScriptInterface {

    public bool $autopublish = true;
    public string $name = 'JT | Social2s Opengraph';
    public string $alias = 'social2sog';
    public string $extension_type = 'plugin';
    public string $plugin_folder = 'system';
    public string $minJtfwVersion = '5.0.0';
    public string $minJtLibsVersion = '5.0.0';
    
    public function install(InstallerAdapter $adapter): bool
    {
        //Log::add('Plugin Jtattach installed correctly.', Log::INFO, 'jerror');
        return true;
    }

    public function update(InstallerAdapter $adapter): bool
    {
        //Log::add('Plugin Jtattach updated correctly.', Log::INFO, 'jerror');
        return true;
    }

    public function uninstall(InstallerAdapter $adapter): bool
    {
        //Log::add('Plugin Jtattach uninstalled correctly.', Log::INFO, 'jerror');
        return true;
    }

    public function preflight(string $type, InstallerAdapter $adapter): bool
    {
        $app = Factory::getApplication();

        if (!$this->jtfwInstalled()) {
            $app->enqueueMessage(Text::_('JT framework required. Please, <a target="_blank" href="https://jtotal.org/libs-fw">download at jtotal.org</a> and install it before.'), 'error');
            return false;
        } else {
            $this->checkOldFrameworkUpdate('https://users.jtotal.org/SOFT/UPDATES/PLUGINS/social2sog/social2sog-update.xml');
            $this->checkUpdaterURL(
                'https://users.jtotal.org/SOFT/UPDATES/PLUGINS/social2sog/social2sog-update.xml'
            );
        }

        return true;
    }

    public function postflight(string $type, InstallerAdapter $adapter): bool
    {
        if ($type === 'install' && $this->autopublish) {
            $this->publishPlugin();
        }

        return true;
    }

    private function jtfwInstalled(): bool
    {
        if (!is_file(JPATH_PLUGINS . '/system/jtframework/autoload.php')) {
            return false;
        }

        $plugin_file = JPATH_PLUGINS . '/system/jtframework/jtframework.xml';
        $xml = $this->parseXMLInstallFile($plugin_file);

        if (!$xml || !isset($xml['version'])) {
            Factory::getApplication()->enqueueMessage("JT framework corrupt. Please, reinstall it.", 'error');
            return false;
        }

        if (version_compare($xml['version'], $this->minJtfwVersion, '<')) {
            Factory::getApplication()->enqueueMessage("JT framework version is too old. Please, update it.", 'error');
            return false;
        }

        return true;
    }

    private function checkUpdaterURL(string $siteToFindAndUpdate): void
    {
        $db = Factory::getDbo();
        
        // Check if site exists
        $query = $db->getQuery(true)
            ->select('COUNT(*) AS count')
            ->from('#__update_sites')
            ->where($db->quoteName('location') . ' = ' . $db->quote($siteToFindAndUpdate));

        $db->setQuery($query);
        $siteExists = $db->loadResult();

        if ($siteExists == 0) {
            // Get extension_id for this plugin
            $query = $db->getQuery(true)
                ->select('extension_id')
                ->from('#__extensions')
                ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'))
                ->where($db->quoteName('element') . ' = ' . $db->quote($this->alias))
                ->where($db->quoteName('folder') . ' = ' . $db->quote($this->plugin_folder));
            $db->setQuery($query);
            $extensionId = $db->loadResult();

            if ($extensionId) {
                // Insert new update site
                $query = $db->getQuery(true)
                    ->insert('#__update_sites')
                    ->columns([$db->quoteName('name'), $db->quoteName('type'), $db->quoteName('location'), $db->quoteName('enabled')])
                    ->values($db->quote($this->name) . ', ' . $db->quote('extension') . ', ' . $db->quote($siteToFindAndUpdate) . ', 1');
                $db->setQuery($query);
                $db->execute();
                
                $updateSiteId = $db->insertid();
                
                // Link update site to extension
                $query = $db->getQuery(true)
                    ->insert('#__update_sites_extensions')
                    ->columns([$db->quoteName('update_site_id'), $db->quoteName('extension_id')])
                    ->values($updateSiteId . ', ' . $extensionId);
                $db->setQuery($query);
                $db->execute();
                
                Factory::getApplication()->enqueueMessage(Text::_($this->name . ' Update Site created :)'), 'info');
            }
        }
    }


    private function checkOldFrameworkUpdate(string $siteToDelete): void
    {
        if (!$siteToDelete) {
            return;
        }

        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('COUNT(*) AS count')
            ->from('#__update_sites')
            ->where($db->quoteName('location') . ' = ' . $db->quote($siteToDelete));

        $db->setQuery($query);
        $result = $db->loadResult();

        if ($result > 0) {
            $query = $db->getQuery(true)
                ->delete($db->quoteName('#__update_sites'))
                ->where($db->quoteName('location') . ' = ' . $db->quote($siteToDelete));
            $db->setQuery($query);
            $db->execute();
            Factory::getApplication()->enqueueMessage(Text::_($this->name . ' OLD Update Sites deleted :)'), 'info');
        }
    }

    private function parseXMLInstallFile(string $file): array|false
    {
        return \Joomla\CMS\Installer\Installer::parseXMLInstallFile($file);
    }

    private function publishPlugin(): void
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->update('#__extensions')
            ->set($db->quoteName('enabled') . ' = 1')
            ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'))
            ->where($db->quoteName('element') . ' = ' . $db->quote($this->alias))
            ->where($db->quoteName('folder') . ' = ' . $db->quote($this->plugin_folder));
        $db->setQuery($query);
        $db->execute();
    }

};
