<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.social2sog // Grupo System, nombre social2sog
 * @copyright   Copyright (C) 2005 - 2023 jtotal.org
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Jtotal\Plugin\System\Social2sOg\Extension; // Namespace según estructura Opción 1

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Filter\OutputFilter;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Registry\Registry;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Router\Route;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\Event\SubscriberInterface;
use Jtotal\Plugin\System\Social2sOg\Helper\S2sDebugOg;
use Jtotal\Plugin\System\Social2sOg\Helper\S2sImagesOg;
use Jtotal\Plugin\System\Social2sOg\Helper\S2sJeventsOg;
use stdClass;

/**
 * Social2s OpenGraph System Plugin.
 * Handles adding OG/Twitter tags once per page load before head compilation.
 *
 * @since  5.3.x
 */
final class Social2sOg extends CMSPlugin implements SubscriberInterface
{
    /**
     * Application object.
     *
     * @var    \Joomla\CMS\Application\CMSApplicationInterface
     * @since  4.0.0? Check your minimum Joomla version
     */
    protected $app;

    /**
     * Load language file automatically on plugin load.
     * @var boolean
     */
    protected $autoloadLanguage = true;

    private ?S2sDebugOg $s2sDebugOg = null;
    private const MAX_DESCRIPTION_LENGTH = 250;

    public static function getSubscribedEvents(): array
    {
        return [
            'onBeforeCompileHead' => 'onBeforeCompileHead',
            'onAfterRender' => 'onAfterRender',
        ];
    }
    

    /**
     * Event listener executed before the HTML head section is compiled.
     * Adds OpenGraph and Twitter Card meta tags if applicable.
     *
     * @return  void
     */
    public function onBeforeCompileHead(): void
    {




        // --- 1. Initial Checks & Setup ---
        $this->app = Factory::getApplication();
        $doc = $this->app->getDocument();
        $input = $this->app->input;

        // Don't run in administrator, non-HTML views, or for non-site requests
        if ($this->app->isClient('administrator') || $doc->getType() !== 'html' || !$this->app->isClient('site')) {
            return;
        }

        $params = $this->params; // Default to this plugin's params

        //Debugger
        $this->s2sDebugOg = new S2sDebugOg($params);
        $this->s2sDebugOg->add('Plugin Started', 'onBeforeCompileHead triggered', 'success', 'play');
        $s2sImagesHelper = null;
        $s2sJeventsHelper = null;
      
        if (version_compare(JVERSION, '5.4', '<')) {
            $this->s2sDebugOg->add('Version Error', 'Requires Joomla 6.0+', 'danger', 'exclamation-triangle');
            return;
        }


        $s2s_og_on = $params->get('opengraph', 1); // OG tags general switch
        $twitter_cards_on = (bool) $params->get('s2s_og_twitter', 0); // Twitter tags specific switch
        if (!$s2s_og_on && !$twitter_cards_on) {
            return; // Both disabled
        }

        try {
            $s2sImagesHelper = new S2sImagesOg($params, $this->s2sDebugOg);
            $s2sJeventsHelper = new S2sJeventsOg($params, $this->s2sDebugOg);
            $this->s2sDebugOg->add('Helpers', 'All helpers instantiated', 'success', 'check');
        } catch (\Throwable $e) {
            $this->s2sDebugOg->add('Helper Error', $e->getMessage(), 'danger', 'times-circle');
        }

        // --- 5. Determine Context and Get Data ---
        $option = $input->getString('option', '');
        $view = $input->getString('view', '');
        $id = $input->getInt('id', 0); // Default 'id', check for specific overrides later
        $itemId = $input->getInt('Itemid', 0);
        $context = $option . '.' . $view;
        $currentAbsoluteUrl = Uri::getInstance()->toString(); // Fallback URL

        $ogData = new \stdClass();
        $ogData->title = $doc->getTitle(); // Start with document title
        $ogData->description = $doc->getMetaData('description'); // Start with document meta desc
        $ogData->images = [];
        $ogData->url = $currentAbsoluteUrl; // Default full link to current page
        $ogData->site_name = Factory::getConfig()->get('sitename');
        $ogData->type = 'website'; // Default type
        $langTag = $this->app->getLanguage()->getTag();

        try {
            switch ($option) {
                case 'com_content':
                    $this->processComContent($id, $view, $ogData, $s2sImagesHelper, $langTag);
                    break;
                
                case 'com_virtuemart':
                    $this->processComVirtuemart($id, $view, $ogData, $s2sImagesHelper, $input);
                    break;
                
                case 'com_jevents':
                    $this->processComJevents($id, $view, $ogData, $s2sImagesHelper, $s2sJeventsHelper, $input, $currentAbsoluteUrl);
                    break;
            }
        } catch (\Exception $e) {
            $this->app->enqueueMessage('Error processing item data for OG: ' . $e->getMessage(), 'error');
            // Allow fallback logic to proceed
        }

        $originalDocTitle = $doc->getTitle();
        if ($ogData->title === $originalDocTitle || $ogData->title === $ogData->site_name) {
            if ($params->get('opengraph_default_data', 0)) {
                $this->app->enqueueMessage('Applying OG Default Data.', 'debug');
                $defaultMode = $params->get('opengraph_default_data', '0');
                $menu = $this->app->getMenu()->getActive();
                $ogData->title = ($defaultMode === '1' && $menu) ? $menu->title : $ogData->site_name;
                if (empty($ogData->description)) $ogData->description = $params->get('opengraph_default_description', $doc->getMetaData('description'));

                // Add default image only if no images were found previously
                if (empty($ogData->images)) {
                    $defaultImg = $params->get('opengraph_default_image', '');
                    if ($defaultImg && $s2sImagesHelper) {
                        $preparedDefault = $s2sImagesHelper->prepareImages([$defaultImg], 1, true);
                        if (!empty($preparedDefault)) $ogData->images = $preparedDefault;
                    } elseif ($defaultImg) { // Fallback if helper failed
                        $cleaned = HTMLHelper::cleanImageURL($defaultImg);
                        if ($cleaned && !empty($cleaned->url)) $ogData->images = [$cleaned->url];
                    }
                }
            } else {
                // No specific data, no default data -> Use basic page info
                $ogData->title = $originalDocTitle ?: $ogData->site_name; // Prefer doc title over sitename
                if (empty($ogData->description)) $ogData->description = $doc->getMetaData('description');
                // No default image handling here if default data is off
            }
             // Ensure URL is still the current page if defaults applied
             $ogData->url = $currentAbsoluteUrl;
             $ogData->type = 'website';
        }
        // Final check on description if still empty
        if (empty($ogData->description)) $ogData->description = $doc->getMetaData('description');


        // --- 8. Write Meta Tags ---
        // Final check for essential data before writing
        if (!empty($ogData->title) && !empty($ogData->url) && strpos($ogData->url, 'http') === 0) {
            // Pass $params for settings checks inside writing methods
            $this->writeOgMetaTags($doc, $params, $ogData, $langTag);
            $this->writeTwitterMetaTags($doc, $params, $ogData);
        } else {
            $this->s2sDebugOg->add('Skipping OG/Twitter tag writing due to missing/invalid title or URL. URL: '.$ogData->url, 'warning');
        }
    } 




    private function processComContent(int $id, string $view, object $ogData, ?S2sImagesOg $s2sImagesHelper, string $langTag): void
    {

        if ($view !== 'article' || $id <= 0) {
            $article = new stdClass();
            $ogData->images = $s2sImagesHelper ? $s2sImagesHelper->getImagesArticle($article) : [];
            return;
        }
        
        $this->s2sDebugOg->add('com_content', 'Processing article ID: ' . $id, 'primary', 'file-text');
        
        if ($view === 'article' && $id > 0) {
            $articleModel = $this->app->bootComponent('com_content')->getMVCFactory()->createModel('Article', 'Site', ['ignore_request' => true]);
            if ($articleModel) {
                // Set state for the model to respect language, access levels etc.
                $articleModel->setState('filter.published', 1);
                $articleModel->setState('filter.language', $langTag);
                // $articleModel->setState('params', $this->app->getParams('com_content')); // Component params
                    // Access checks are usually handled by the model if state is set correctly
                // $user = $this->app->getIdentity();
                // $articleModel->setState('filter.access', true); // Let model check access
                $articleModel->setState('params', $this->app->getParams()); // Pass application parameters
                $article = $articleModel->getItem($id); // Fetch the specific item

                if (is_object($article) && !empty($article->id)) {
                    $ogData->title = $article->title;
                    $ogData->description = $this->_prepareDescription($article->introtext ?? '', $article->metadesc ?? $ogData->description); // Use article meta first

                    $ogData->images = $s2sImagesHelper ? $s2sImagesHelper->getImagesArticle($article) : [];

                    $ogData->type = 'article';
                    $slug = $article->alias ?? OutputFilter::stringURLSafe($article->title);
                    $catid = $article->catid ?? 0;
                    if ($slug && $catid) {
                        $nonSefUrl = 'index.php?option=com_content&view=article&id=' . $slug . '&catid=' . $catid;
                        $ogData->url = Route::_($nonSefUrl, false, 2);
                    }
                }
            }
        }
    }


    private function processComJevents(int $id, string $view, object $ogData, ?S2sImagesOg $s2sImagesHelper, ?S2sJeventsOg $s2sJeventsHelper, $input, string $currentAbsoluteUrl): void
    {

        $id = $input->getInt('evid', $id);
        if ($view !== 'icalrepeat' || $id <= 0) {
            return;
        }
      
        if ($view === 'icalrepeat' && $id > 0) {
            $this->s2sDebugOg->add('JEvents Detection', 'Processing event ID: ' . $id, 'primary', 'calendar');
            
            // Cargar evento usando el helper
            $event = $s2sJeventsHelper ? $s2sJeventsHelper->loadJEventsEvent($id) : null;
            


            if ($event && is_object($event)) {
                // Obtener datos estructurados
                $eventData = $s2sJeventsHelper->getEventData($event);

                
                
                $ogData->title = $eventData->title ?: $ogData->title;
                $ogData->description = $this->_prepareDescription(
                    $eventData->description,
                    $event->metadesc ?? $ogData->description
                );
                
                // Obtener imágenes
                $ogData->images = $s2sImagesHelper ? $s2sImagesHelper->getImagesJEvents($eventData) : [];

                foreach ($ogData->images as $imgUrl) {
                    $this->s2sDebugOg->add('JEvents Image', 'Found image: ' . $imgUrl . S2sImagesOg::getThumbImg($imgUrl), 'info', 'image');
                }
                
                $ogData->type = 'event';
                $ogData->url = $s2sJeventsHelper ?
                    ($s2sJeventsHelper->getJEventsUrl($event) ?: $currentAbsoluteUrl) :
                    $currentAbsoluteUrl;
                
                // Agregar metadatos de evento para OpenGraph
                if (!empty($eventData->startDate)) {
                    $ogData->event_start_time = date('c', strtotime($eventData->startDate));
                }
                if (!empty($eventData->endDate)) {
                    $ogData->event_end_time = date('c', strtotime($eventData->endDate));
                }
                
                $this->s2sDebugOg->add('JEvents Complete', 'Event processed successfully', 'success', 'check-circle');
            } else {
                $this->s2sDebugOg->add('JEvents Error', 'Event could not be loaded', 'danger', 'times');
            }
        }
    }

    private function processComVirtuemart(int $id, string $view, object $ogData, ?S2sImagesOg $imagesHelper, $input): void
    {
        if ($view === 'productdetails') {
            $vmId = $input->getInt('virtuemart_product_id', $id); // Use specific VM param
            if ($vmId > 0 && \Joomla\CMS\Component\ComponentHelper::isEnabled('com_virtuemart')) {
                // Load VM classes and model (ensure paths/classes are correct for J5/VM version)
                if (!class_exists('VmConfig')) { /* ... load VmConfig ... */ }
                $vmModelPath = VMPATH_SITE.'/models/productdetails.php'; // Use VM constant if available
                if (!class_exists('VirtueMartModelProductdetails') && file_exists($vmModelPath)) require_once($vmModelPath);

                if (class_exists('VirtueMartModelProductdetails')) {
                    try {
                        $productModel = new \VirtueMartModelProductdetails();
                        // VM Model usually gets ID from input, ensure 'virtuemart_product_id' is correct
                        $product = $productModel->getProduct(false); // Pass false to maybe avoid triggering extra events? Check VM API.
                        if ($product && is_object($product) && !empty($product->virtuemart_product_id)) {
                            $ogData->title = $product->product_name ?? $ogData->title;
                            $ogData->description = $this->_prepareDescription($product->product_desc ?? '', $product->metadesc ?? $ogData->description);
                            $ogData->images = $s2sImagesHelper ? $s2sImagesHelper->getImagesVirtuemart($product) : [];
                            $ogData->type = 'product';
                            $ogData->url = $currentAbsoluteUrl; // Use current URL for VM safety
                            // Add product:price:amount, product:price:currency etc. if desired
                        }
                    } catch (\Exception $eVm) { /* Log error */ }
                }
            }
        }
    }

    /** Prepara descripción: limpia HTML, limita longitud */
    private function _prepareDescription($primaryText = '', $fallbackMeta = ''): string {
        $desc = '';
        if (!empty($primaryText)) {
            // Limpieza básica - puede necesitar más si hay HTML complejo
            $desc = strip_tags(html_entity_decode($primaryText, ENT_QUOTES | ENT_HTML5, 'UTF-8'));
            $desc = str_replace(["\r", "\n", "\r\n"], ' ', $desc);
            $desc = trim(preg_replace('/\s{2,}/', ' ', $desc)); // Reemplaza espacios múltiples
            $maxLength = self::MAX_DESCRIPTION_LENGTH;
            if (mb_strlen($desc) > $maxLength) {
                $desc = mb_substr($desc, 0, $maxLength - 3) . '...';
            }
        }
        // Usa fallback si la descripción primaria está vacía después de limpiar
        if (empty($desc) && !empty($fallbackMeta)) {
            $desc = trim(strip_tags(html_entity_decode($fallbackMeta, ENT_QUOTES | ENT_HTML5, 'UTF-8')));
            // Podrías aplicar límite de longitud al fallback también si quieres
        }
        return $desc;
    }


    /** Escribe etiquetas Open Graph */
    private function writeOgMetaTags(object $doc, Registry $params, object $ogData, string $langTag): void
    {
        if (!$params->get('opengraph', 1)) return; // Check general OG switch

        $tags = '';
        // Usa htmlspecialchars para asegurar que el contenido del atributo sea válido
        if (!empty($ogData->title)) {
            $tags .= '<meta property="og:title" content="' 
                . htmlspecialchars($ogData->title, ENT_QUOTES, 'UTF-8') 
                . '">' . "\n";
        }

        if (!empty($ogData->description)) {
            $tags .= '<meta property="og:description" content="' 
                . htmlspecialchars($ogData->description, ENT_QUOTES, 'UTF-8') 
                . '">' . "\n";
        }

        if (!empty($ogData->url)) {
            $tags .= '<meta property="og:url" content="' 
                . htmlspecialchars($ogData->url, ENT_QUOTES, 'UTF-8') 
                . '">' . "\n";
        }

        if (!empty($ogData->site_name)) {
            $tags .= '<meta property="og:site_name" content="' 
                . htmlspecialchars($ogData->site_name, ENT_QUOTES, 'UTF-8') 
                . '">' . "\n";
        }

        if (!empty($ogData->type)) {
            $tags .= '<meta property="og:type" content="' 
                . htmlspecialchars($ogData->type, ENT_QUOTES, 'UTF-8') 
                . '">' . "\n";
        }

        if (!empty($langTag)) {
            $tags .= '<meta property="og:locale" content="' 
                . htmlspecialchars(str_replace('-', '_', $langTag), ENT_QUOTES, 'UTF-8') 
                . '">' . "\n";
        }

        if (!empty($ogData->images)) {
            $tags .= $this->_generateImageMetaTags($ogData->images, 'og');
        }

        // App ID
        $appId = $params->get('opengraph_custom_app_id_opt', 0) == 1 ? $params->get('opengraph_custom_app_id', '') : '514279921989553';
        if (!empty($appId)) $tags .= '<meta property="fb:app_id" content="' . htmlspecialchars($appId, ENT_QUOTES, 'UTF-8') . '">' . "\n";

        // Add other specific tags based on ogData type if needed (e.g., article:published_time)
        if ($ogData->type === 'article') {
             // Ejemplo: Añadir tiempo de publicación si existe en $ogData->published_time
             // if (!empty($ogData->published_time)) $tags .= '<meta property="article:published_time" content="' . htmlspecialchars($ogData->published_time) . '">'."\n";
        }
         if ($ogData->type === 'product') {
             // Ejemplo: Añadir precio si existe en $ogData->price, $ogData->currency
             // if (!empty($ogData->price) && !empty($ogData->currency)) {
             //      $tags .= '<meta property="product:price:amount" content="' . htmlspecialchars($ogData->price) . '">'."\n";
             //      $tags .= '<meta property="product:price:currency" content="' . htmlspecialchars($ogData->currency) . '">'."\n";
             // }
        }


        if (!empty($tags)) {
            $doc->addCustomTag($tags);
            $tagsDebug = str_replace("<meta", '<br><strong>meta</strong>', $tags);
            $this->s2sDebugOg->add('Added OG Tags', '<small style="max-width:400px">' . $tagsDebug . '</small>', 'success', 'tags');
        }
    }

    /** Escribe etiquetas Twitter Card */
    private function writeTwitterMetaTags(object $doc, Registry $params, object $ogData): void
    {
        if (!(bool)$params->get('s2s_og_twitter', 0)) return; // Check Twitter specific switch
        $twitterUser = $params->get('s2s_og_twitter_user', '');
        if (empty($twitterUser)) return; // Twitter user required
        $twitterUser = '@' . ltrim($twitterUser, '@');

        $tags = '';
        $cardType = $params->get('s2s_og_twitter_summary', 0) ? 'summary_large_image' : 'summary';
        $tags .= '<meta name="twitter:card" content="' . $cardType . '">' . "\n";
        $tags .= '<meta name="twitter:site" content="' . htmlspecialchars($twitterUser, ENT_QUOTES, 'UTF-8') . '">' . "\n";
        if (!empty($ogData->title)) $tags .= '<meta name="twitter:title" content="' . htmlspecialchars($ogData->title, ENT_QUOTES, 'UTF-8') . '">' . "\n";
        if (!empty($ogData->description)) {
            $twitterDesc = mb_substr($ogData->description, 0, 197); // Truncate for Twitter
            if(mb_strlen($ogData->description) > 197) $twitterDesc .= '...';
            $tags .= '<meta name="twitter:description" content="' . htmlspecialchars($twitterDesc, ENT_QUOTES, 'UTF-8') . '">' . "\n";
        }
        // Image (first one - generateImageMetaTags handles array check)
        if (!empty($ogData->images)) {
            $tags .= $this->_generateImageMetaTags($ogData->images, 'twitter', $ogData->title ?? '');
        }

        if (!empty($tags)) {
            $doc->addCustomTag($tags);
        }
    }

    /**
     * Genera las meta etiquetas para imágenes (OG o Twitter).
     * Asume que $imageUrls es un array de URLs ABSOLUTAS.
     *
     * @param array  $imageUrls Array of absolute image URLs.
     * @param string $type      'og' or 'twitter'.
     * @return string The generated meta tag string.
    */
    private function _generateImageMetaTags(array $imageUrls, string $type = 'og', string $altText = ''): string
    {
        if (empty($imageUrls)) {
            return '';
        }

        $tags = '';
        $imagesToProcess = ($type === 'og') ? $imageUrls : array_slice($imageUrls, 0, 1);

        foreach ($imagesToProcess as $absoluteUrl) {
            if (!is_string($absoluteUrl) || empty($absoluteUrl) || strpos($absoluteUrl, 'http') !== 0) {
                $this->s2sDebugOg->add('Skipping invalid image URL for meta tag: ' . print_r($absoluteUrl, true), 'warning');
                continue;
            }

            $escapedUrl = htmlspecialchars($absoluteUrl, ENT_QUOTES, 'UTF-8');

            if ($type === 'og') {
                $tags .= '<meta property="og:image" content="' . $escapedUrl . '">' . "\n";
                $secureUrl = (strpos($absoluteUrl, 'https://') === 0) ? $escapedUrl : $escapedUrl;
                $tags .= '<meta property="og:image:secure_url" content="' . $secureUrl . '">' . "\n";
                $imageInfo = S2sImagesOg::getImageInfo($absoluteUrl);
                if ($imageInfo !== false && isset($imageInfo[0], $imageInfo[1], $imageInfo['mime'])) {
                    $tags .= '<meta property="og:image:type" content="' . htmlspecialchars($imageInfo['mime']) . '">' . "\n";
                    $tags .= '<meta property="og:image:width" content="' . ((int)$imageInfo[0]) . '">' . "\n";
                    $tags .= '<meta property="og:image:height" content="' . ((int)$imageInfo[1]) . '">' . "\n";
                }

                




            } elseif ($type === 'twitter') {
                $tags .= '<meta name="twitter:image" content="' . $escapedUrl . '">' . "\n";
                $tags .= '<meta name="twitter:image:alt" content="' . htmlspecialchars($altText ?: 'Image', ENT_QUOTES, 'UTF-8') . '">' . "\n";
            }
        }

        return $tags;
    }

    public function onAfterRender(): void
    {
        // Solo mostrar si existe debugger y está habilitado
        if (!$this->s2sDebugOg || !$this->s2sDebugOg->isEnabled()) {
            return;
        }

        if (!$this->app->isClient('site')) {
            return;
        }
                       
        $body = $this->app->getBody();
       
        if (empty($body)) {
            return;
        }

        // Capturar salida del debugger
        ob_start();
        $this->s2sDebugOg->show();
        $debugHtml = ob_get_clean();

        // Inyectar antes del </body>
        $body = str_replace('</body>', $debugHtml . '</body>', $body);
        
        $this->app->setBody($body);
    }

} // End class
