<?php

namespace Jtotal\Plugin\System\Social2sOg\Helper;

use Joomla\CMS\Factory;
use Joomla\Registry\Registry;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Uri\Uri;
use Joomla\Filesystem\File;
use Jtotal\Plugin\Content\Social2s\Extension\Social2s;

defined('_JEXEC') or die;

/**
 * Helper class for extracting and preparing images for Social2sOg System plugin.
 */
class S2sImagesOg
{
    protected Registry $params;
    private $app;
    private ?S2sDebugOg $s2sDebugOg = null;
    private array $imageSizeCache = [];

    public function __construct(Registry $params, ?S2sDebugOg $s2sDebugOg = null)
    {
        $this->params = $params;
        $this->app = Factory::getApplication();
        $this->s2sDebugOg = $s2sDebugOg ?: new S2sDebugOg($params);
    }

    /**
     * Get images from a standard Joomla com_content article object.
     */
    public function getImagesArticle(object $article): array
    {
        $alwaysImages = $this->prepareDefAlways();

        if ($alwaysImages !== false) {
            return $alwaysImages;
        }

        $candidateImages = [];
        $isSingleImageMode = ($this->params->get('og_multi_img', '0') == '0');

        if (isset($article->jcfields) && !empty($article->jcfields['og-image']->rawvalue)) {
            $candidateImages[] = $article->jcfields['og-image']->rawvalue;
        }

        if ($isSingleImageMode && !empty($candidateImages)) {
            // Se procesará al final
        } else {
            if (!empty($article->images)) {
                try {
                    $joomlaImages = json_decode($article->images, false, 512, JSON_THROW_ON_ERROR);
                    if (!empty($joomlaImages->image_fulltext)) {
                        $candidateImages[] = $joomlaImages->image_fulltext;
                        $this->s2sDebugOg->add('Full', ' ' . $joomlaImages->image_fulltext . (self::getThumbImg($joomlaImages->image_fulltext) ?: ''), 'success', 'check');
                    }
                    if (!$this->params->get('og_skip_intro_img', '0') && !empty($joomlaImages->image_intro)) {
                        $candidateImages[] = $joomlaImages->image_intro;
                        $this->s2sDebugOg->add('Intro', ' ' . $joomlaImages->image_intro . (self::getThumbImg($joomlaImages->image_intro) ?: ''), 'success', 'check');
                    }
                } catch (\JsonException $e) {
                    $this->s2sDebugOg->add('Error', 'Error decoding article images JSON: ' . $e->getMessage(), 'danger', 'exclamation-triangle');
                }
            }

            $og_add_dom_img = $this->params->get('og_add_dom_img', '0');
            if ($og_add_dom_img !== '0' && isset($article->introtext)) {
                $fullText = $article->introtext . ($article->fulltext ?? '');
                $findAllInDom = !$isSingleImageMode && ($og_add_dom_img === '2');
                $dom_images = $this->getDomImage($fullText, $findAllInDom);
                if (!empty($dom_images)) {
                    foreach ($dom_images as $img) {
                        $this->s2sDebugOg->add('Article Image', 'Prepared image: ' . $img, 'success', 'check');
                    }

                    $candidateImages = array_merge($candidateImages, $dom_images);
                }
            }
        }

        $preparedImages = $this->prepareImages($candidateImages);

        if (empty($preparedImages)) {
            if ($this->params->get('opengraph_default_image_opt', '0') == '3') {
                $defaultImg = $this->params->get('opengraph_default_image', '');
                if (!empty($defaultImg)) {
                    $preparedImages = $this->prepareImages([$defaultImg], 1, true);
                }
            }
        }
        

        return $preparedImages;
    }

    public function getImagesVirtuemart(object $product): array {
        $images = [];
        $always = $this->prepareDefAlways();
        if ($always !== false) return $always;
        $max = ($this->params->get('og_multi_img', '0') == '0') ? 1 : (int)$this->params->get('og_max_multi_img', 3);

        if (!empty($product->images) && is_array($product->images)) {
            foreach($product->images as $image) {
                if (count($images) >= $max) break;
                if (isset($image->file_url) && !empty($image->file_url)) {
                    $images[] = $image->file_url;
                }
            }
        }

        $og_add_dom_img = $this->params->get('og_add_dom_img', '0');
        if ($og_add_dom_img !== '0' && count($images) < $max && isset($product->product_desc)) {
            $dom_images = $this->getDomImage($product->product_desc, ($og_add_dom_img == '2'));
            $needed = $max - count($images);
            if ($needed > 0) $images = array_merge($images, array_slice($dom_images, 0, $needed));
        }

        $images_ok = $this->prepareImages($images, $max);
        return $images_ok;
    }

    /**
     * Get images from a K2 item object.
     */
    public function getImagesK2(object $item): array
    {
        $always = $this->prepareDefAlways();
        if ($always !== false) {
            return $always;
        }

        $candidateImages = [];
        $isSingleImageMode = ($this->params->get('og_multi_img', '0') == '0');

        $imageIdHash = md5("Image" . $item->id);
        $cachePathXL = JPATH_SITE . '/media/k2/items/cache/' . $imageIdHash . '_XL.jpg';
        $cachePathGeneric = JPATH_SITE . '/media/k2/items/cache/' . $imageIdHash . '_Generic.jpg';

        if (File::exists($cachePathXL)) {
            $candidateImages[] = 'media/k2/items/cache/' . $imageIdHash . '_XL.jpg';
        } elseif (File::exists($cachePathGeneric)) {
            $candidateImages[] = 'media/k2/items/cache/' . $imageIdHash . '_Generic.jpg';
        }

        if (!$isSingleImageMode || empty($candidateImages)) {
            $og_add_dom_img = $this->params->get('og_add_dom_img', '0');
            if ($og_add_dom_img !== '0' && isset($item->introtext)) {
                $findAllInDom = !$isSingleImageMode && ($og_add_dom_img === '2');
                $dom_images = $this->getDomImage($item->introtext, $findAllInDom);
                if (!empty($dom_images)) {
                    $candidateImages = array_merge($candidateImages, $dom_images);
                }
            }
        }

        $preparedImages = $this->prepareImages($candidateImages);

        if (empty($preparedImages)) {
            if ($this->params->get('opengraph_default_image_opt', '0') == '3') {
                $defaultImg = $this->params->get('opengraph_default_image', '');
                if (!empty($defaultImg)) {
                    $preparedImages = $this->prepareImages([$defaultImg], 1, true);
                }
            }
        }

        return $preparedImages;
    }

    /**
     * Get images from a jEvents event object.
     */
    public function getImagesJevents(object $event): array
    {
        $alwaysImages = $this->prepareDefAlways();

        if ($alwaysImages !== false) {
            return $alwaysImages;
        }

        $candidateImages = [];
        $isSingleImageMode = ($this->params->get('og_multi_img', '0') == '0');

        $og_add_dom_img = $this->params->get('og_add_dom_img', '0');

        if ($og_add_dom_img !== '0' && isset($event->description)) {

            $findAllInDom = !$isSingleImageMode && ($og_add_dom_img === '2');
            $dom_images = $this->getDomImage($event->description, $findAllInDom);
            if (!empty($dom_images)) {
                $candidateImages = $dom_images;
            }
        }

        $preparedImages = $this->prepareImages($candidateImages);

        if (empty($preparedImages)) {
            if ($this->params->get('opengraph_default_image_opt', '0') == '3') {
                $defaultImg = $this->params->get('opengraph_default_image', '');
                if (!empty($defaultImg)) {
                    $preparedImages = $this->prepareImages([$defaultImg], 1, true);
                }
            }
        }

        return $preparedImages;
    }

private function prepareImages(array $rawImageUrls, ?int $maxImages = null, bool $skipSizeCheck = false): array 
{
    if ($maxImages === null) {
        $maxImages = ($this->params->get('og_multi_img', '0') == '0') ? 1 : (int) $this->params->get('og_max_multi_img', 3);
    }

    $skipSizeCheck = $this->params->get('og_skip_min_img', '0') === '0' ? true : false;

    $images_ok = [];
    
    foreach ($rawImageUrls as $image) {
        // Validar que es string y no vacío
        if (!is_string($image) || empty(trim($image))) {
            $this->s2sDebugOg->add('prepareImages', 'Image rejected: not string or empty', 'warning', 'exclamation-triangle');
            continue;
        }
        
        $cleanedImage = HTMLHelper::cleanImageURL($image);
        
        if ($cleanedImage && !empty($cleanedImage->url)) {
            $absoluteUrl = $cleanedImage->url;
            
            // Convertir a URL absoluta si es necesario
            if (strpos($absoluteUrl, 'http') !== 0) {
                $absoluteUrl = Uri::base() . ltrim($absoluteUrl, '/');
                
                // Validar que ahora sí es absoluta
                if (strpos($absoluteUrl, 'http') !== 0) {
                    $this->s2sDebugOg->add('prepareImages', 'URL rejected: not absolute after conversion', 'warning', 'exclamation-triangle');
                    continue;
                }
            }
            
            // Validar tamaño
            if ($skipSizeCheck) {
                $images_ok[] = $absoluteUrl;
            } else {
                $sizeValid = $this->imgSize($absoluteUrl);
                
                if ($sizeValid) {
                    $images_ok[] = $absoluteUrl;
                } else {
                    $this->s2sDebugOg->add('prepareImages', 'Image rejected by size: ' . $image, 'danger', 'times');
                }
            }
        } else {
            $this->s2sDebugOg->add('prepareImages', 'cleanImageURL failed for: ' . $image, 'warning', 'exclamation-triangle');
        }
    }

    $this->s2sDebugOg->add('prepareImages', 'Images accepted: ' . count($images_ok) . ' of ' . count($rawImageUrls), 'info', 'images');

    $images_unique = array_unique($images_ok);
    $images_final = array_values($images_unique);
    
    if ($maxImages > 0 && count($images_final) > $maxImages) {
        $images_final = array_slice($images_final, 0, $maxImages);
    }

    return $images_final;
}

private function imgSize(string $imgUrl): bool
{
    $minWidth = (int) $this->params->get('og_img_min_width', 200);
    $minHeight = (int) $this->params->get('og_img_min_height', 200);
    
    // Si ambos son 0 o negativos, no validar
    if ($minWidth <= 0 && $minHeight <= 0) {
        return true;
    }
    
    // Validar que sea URL absoluta
    if (strpos($imgUrl, 'http') !== 0) {
        $this->s2sDebugOg->add('imgSize', 'URL is not absolute: ' . $imgUrl, 'danger', 'times');
        return false;
    }

    try {
        $imageInfo = $this->getImageInfo($imgUrl);
        
        if ($imageInfo === false || !isset($imageInfo[0], $imageInfo[1])) {
            $this->s2sDebugOg->add('imgSize', 'Failed to get image info: ' . $imgUrl, 'danger', 'times');
            return false;
        }
        
        $width = (int) $imageInfo[0];
        $height = (int) $imageInfo[1];
        
        $valid = ($minWidth <= 0 || $width >= $minWidth) && ($minHeight <= 0 || $height >= $minHeight);
        
        if (!$valid) {
            $this->s2sDebugOg->add('imgSize', "Dimensions {$width}x{$height} < required {$minWidth}x{$minHeight}", 'warning', 'exclamation-triangle');
        }
        
        return $valid;

    } catch (\Exception $e) {
        $this->s2sDebugOg->add('imgSize', 'Exception: ' . $e->getMessage(), 'danger', 'exclamation-triangle');
        return false;
    }
}




    public static function getImageInfo($imgUrl)
    {

        $cacheKey = 'og_img_size_' . md5($imgUrl);
        $cache = Factory::getContainer()->get(\Joomla\CMS\Cache\CacheControllerFactoryInterface::class)->createCacheController('output', ['defaultgroup' => 'com_social2sog']);
        
        $params = Factory::getApplication()->getParams('plg_system_social2sog');
        if ($params->get('s2sog_force_cache', '0') === '1') {
            $cache->options['caching'] = true;
        }

        //no cache
        if ($cache->options['caching'] !== true) {
            $imageInfo = @getimagesize($imgUrl);
        } else {
            $imageInfo = $cache->get($cacheKey);

            if ($imageInfo !== false) {
                // Ya está en caché
            } else {
                // Intentar obtener dimensiones
                $imageInfo = @getimagesize($imgUrl);
                
                if ($imageInfo === false || !isset($imageInfo[0], $imageInfo[1])) {
                    // Cachear el fallo también para no reintentar
                    $cache->store(false, $cacheKey);
                    return false;
                }
                // Guardar en cache de Joomla
                $cache->store($imageInfo, $cacheKey);
            }
        }

        return $imageInfo;
    }



    //TODO
    private function imgSizeApcu(string $imgUrl): bool
    {
        $minWidth = (int) $this->params->get('og_img_min_width', 200);
        $minHeight = (int) $this->params->get('og_img_min_height', 200);
        
        if ($minWidth <= 0 && $minHeight <= 0) {
            return true;
        }
        
        if (strpos($imgUrl, 'http') !== 0) {
            return false;
        }

        // Cache key único
        $cacheKey = 'og_img_size_' . md5($imgUrl);
        
        // Intentar obtener de cache (APCu si está disponible)
        $imageInfo = false;
        if (function_exists('apcu_fetch')) {
            $imageInfo = apcu_fetch($cacheKey);
        }
       
        // Si no está en cache, obtener
        if ($imageInfo === false) {
            $imageInfo = @getimagesize($imgUrl);
            
            if ($imageInfo === false || !isset($imageInfo[0], $imageInfo[1])) {
                return false;
            }
            
            // Guardar en cache por 24 horas
            if (function_exists('apcu_store')) {
                apcu_store($cacheKey, $imageInfo, 86400);
            }
        }

        $width = (int) $imageInfo[0];
        $height = (int) $imageInfo[1];
        
        return ($minWidth <= 0 || $width >= $minWidth) 
            && ($minHeight <= 0 || $height >= $minHeight);
    }


    /*
    // Extract images from HTML content using DOMDocument (slower method)
    private function getDomImage(string $html, bool $extractAll = false): array {
        $images = [];
        if (empty(trim($html))) return $images;

        $domdoc = new \DOMDocument();
        libxml_use_internal_errors(true);
        @$domdoc->loadHTML($html);
        libxml_clear_errors();

        $tags = $domdoc->getElementsByTagName('img');
        foreach ($tags as $tag) {
            if ($tag->hasAttribute('src')) {
                $imageUrl = $tag->getAttribute('src');
                if (empty($imageUrl) || strpos($imageUrl, 'data:image') === 0) continue;
                $images[] = $imageUrl;
                if (!$extractAll) break;
            }
        }

        return $images;
    }*/


    private function getDomImage(string $html, bool $extractAll = false): array
    {
        $images = [];
        
        // Early return si HTML vacío
        if (empty(trim($html))) {
            return $images;
        }

        // Regex optimizado para capturar src con diferentes formatos de comillas
        $pattern = '/<img[^>]+src\s*=\s*(["\'])([^"\']+)\1[^>]*>/i';
        
        if (preg_match_all($pattern, $html, $matches, PREG_SET_ORDER)) {
            foreach ($matches as $match) {
                $imageUrl = $match[2]; // El segundo grupo captura la URL
                
                // Filtrar URLs no válidas
                if (empty($imageUrl) ||
                    strpos($imageUrl, 'data:image') === 0 ||
                    strpos($imageUrl, 'data:application') === 0) {
                    continue;
                }
                
                // Decodificar entidades HTML si es necesario
                $imageUrl = html_entity_decode($imageUrl, ENT_QUOTES | ENT_HTML5, 'UTF-8');
                
                $images[] = trim($imageUrl);
                
                if (!$extractAll) {
                    break;
                }
            }
        }

        return $images;
    }



    /*
        * Prepare images from default always settings
    */
    private function prepareDefAlways(): array|false {
        //PREPARE IMAGES FROM DEFAULT ALWAYS SETTINGS
        $defaultImg = $this->params->get('opengraph_default_image', '');
        $defaultImgOpt = $this->params->get('opengraph_default_image_opt', '0');

        // Debug info
        if ($defaultImgOpt == '0') {
            $this->s2sDebugOg->add('Default image', 'Never use Default image', 'default', 'message');
        } else if ($defaultImgOpt == '1') {
            $this->s2sDebugOg->add('Default image', 'Always use Default image', 'warning', 'exclamation-triangle');
            if (!empty($defaultImg)) {
                $preparedDefault = $this->prepareImages([$defaultImg], 1, true);
                $this->s2sDebugOg->add('Default image', 'Default image defined: ' . $preparedDefault[0] . (self::getThumbImg($preparedDefault[0]) ?: ''), 'success', 'check');
            } else {
                 $this->s2sDebugOg->add('Default image', 'No image defined', 'danger', 'image');
            }
        } else if ($defaultImgOpt == '3') {
            $this->s2sDebugOg->add('Default image', 'Use Default image if no other image found', 'info', 'info-circle');
            if (!empty($defaultImg)) {
                $preparedDefault = $this->prepareImages([$defaultImg], 1, true);
                $this->s2sDebugOg->add('Default image', 'Default image defined: ' . $preparedDefault[0] . (self::getThumbImg($preparedDefault[0]) ?: ''), 'success', 'check');
            } else {
                $this->s2sDebugOg->add('Default image', 'No image defined', 'danger', 'image');
            }
        }

        if ($defaultImgOpt == '1') {
            if (!empty($defaultImg)) {
                $preparedDefault = $this->prepareImages([$defaultImg], 1, true);
                if (!empty($preparedDefault)) {
                    return $preparedDefault;
                }
            }
        }
        return false;
    }

    public static function getThumbImg(string $imgUrl): string
    {
        $image = '<img src="' . htmlspecialchars($imgUrl, ENT_QUOTES, 'UTF-8') . '" alt="OG Image" style="max-width: 64px; max-height: 64px; border: 1px solid #ccc; padding: 2px;" />';
        return $image;
    }

}
