<?php

namespace Jtotal\Plugin\System\Social2sOg\Helper;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use Joomla\Registry\Registry;

/**
 * Helper class for JEvents integration (JEvents 3.6+)
 */
class S2sJeventsOg
{
    protected Registry $params;
    private $app;
    private ?S2sDebugOg $s2sDebugOg = null;

    public function __construct(Registry $params, ?S2sDebugOg $debugger = null)
    {
        $this->params = $params;
        $this->app = Factory::getApplication();
        $this->s2sDebugOg = $debugger;
    }

    /**
     * Cargar evento de JEvents 3.6+ (Joomla 5 compatible)
     */
    public function loadJEventsEvent(int $eventId): ?object
    {
        // Verificar que JEvents esté instalado y habilitado
        if (!ComponentHelper::isEnabled('com_jevents')) {
            $this->debug('JEvents', 'Component not enabled', 'danger', 'times-circle');
            return null;
        }

        try {
            $this->debug('JEvents', 'Loading event ID: ' . $eventId, 'info', 'search');

            $event = $this->loadEventViaMVC($eventId);
            
            if ($event) {
                $this->debug('JEvents MVC', 'Event loaded successfully', 'success', 'check-circle');
                return $event;
            }

            $event = $this->loadEventViaDatabase($eventId);
            
            if ($event) {
                $this->debug('JEvents DB', 'Event loaded via database', 'success', 'database');
                return $event;
            }

            $this->debug('JEvents', 'Event not found', 'warning', 'exclamation-triangle');
            return null;
        } catch (\Throwable $e) {
            $this->debug('JEvents Exception', $e->getMessage(), 'danger', 'bug');
            return null;
        }
    }

    /**
     * Cargar evento usando el MVC de Joomla 5
     */
    private function loadEventViaMVC(int $eventId): ?object
    {
        try {
            // Cargar el componente JEvents
            $component = $this->app->bootComponent('com_jevents');
            
            if (!$component) {
                $this->debug('JEvents MVC', 'Failed to boot component', 'warning', 'power-off');
                return null;
            }

            // Obtener la MVC Factory
            $mvcFactory = $component->getMVCFactory();
            
            if (!$mvcFactory instanceof MVCFactoryInterface) {
                $this->debug('JEvents MVC', 'Invalid MVC Factory', 'warning', 'cogs');
                return null;
            }

            // Crear el modelo de evento
            $model = $mvcFactory->createModel('Event', 'Site', ['ignore_request' => true]);
            
            if (!$model) {
                $this->debug('JEvents MVC', 'Model not found', 'warning', 'file-o');
                return null;
            }

            // Cargar el evento
            $event = $model->getItem($eventId);
            
            if (!$event || !is_object($event)) {
                return null;
            }

            // Verificar que tenga datos válidos
            if (empty($event->ev_id) && empty($event->rp_id) && empty($event->id)) {
                return null;
            }

            return $event;

        } catch (\Throwable $e) {
            $this->debug('JEvents MVC Error', $e->getMessage(), 'warning', 'exclamation');
            return null;
        }
    }

    /**
     * Cargar evento directamente de la base de datos (fallback)
     */
    private function loadEventViaDatabase(int $eventId): ?object
    {
        try {
            $db = Factory::getContainer()->get('DatabaseDriver');
            
            // JEvents usa dos tablas principales:
            // #__jevents_vevent (evento base)
            // #__jevents_repetition (repeticiones del evento)
            
            $query = $db->getQuery(true)
                ->select('v.*')
                ->select('r.rp_id, r.startrepeat, r.endrepeat')
                ->from($db->quoteName('#__jevents_repetition', 'r'))
                ->join('LEFT', $db->quoteName('#__jevents_vevent', 'v') . ' ON r.eventid = v.ev_id')
                ->where($db->quoteName('r.rp_id') . ' = ' . (int) $eventId);
            
            $db->setQuery($query);
            $event = $db->loadObject();
            
            if (!$event) {
                // Intentar cargar solo del evento base
                $query = $db->getQuery(true)
                    ->select('*')
                    ->from($db->quoteName('#__jevents_vevent'))
                    ->where($db->quoteName('ev_id') . ' = ' . (int) $eventId);
                
                $db->setQuery($query);
                $event = $db->loadObject();
            }
            
            return $event ?: null;

        } catch (\Exception $e) {
            $this->debug('JEvents DB Error', $e->getMessage(), 'danger', 'database');
            return null;
        }
    }

    /**
     * Obtener URL del evento JEvents
     */
    public function getJEventsUrl(object $event): ?string
    {
        try {
            // Determinar ID correcto (puede ser rp_id o ev_id)
            $eventId = $event->rp_id ?? $event->ev_id ?? $event->id ?? 0;
            
            if ($eventId <= 0) {
                $this->debug('JEvents URL', 'Invalid event ID', 'warning', 'exclamation');
                return null;
            }

            // Construir URL SEF
            $nonSefUrl = 'index.php?option=com_jevents&view=icalrepeat&id=' . $eventId;
            
            // Añadir Itemid si está disponible
            $menu = $this->app->getMenu();
            $items = $menu->getItems('component', 'com_jevents');
            
            if (!empty($items)) {
                $nonSefUrl .= '&Itemid=' . $items[0]->id;
            }
            
            $sefUrl = Route::_($nonSefUrl, false, 2);
            
            $this->debug('JEvents URL', 'Generated: ' . basename($sefUrl), 'success', 'link');
            
            return $sefUrl;

        } catch (\Throwable $e) {
            $this->debug('JEvents URL Error', $e->getMessage(), 'warning', 'chain-broken');
            return null;
        }
    }

    public function getEventData(object $event): object
    {
        $data = new \stdClass();
        
        // Parsear rawdata de forma segura
        $rawdata = [];
        if (!empty($event->rawdata)) {
            $rawdata = $this->parseEventRawdata($event->rawdata);
        }
        
        // Título
        $data->title = $event->summary ??
                    $event->title ??
                    $rawdata['SUMMARY'] ??
                    '';
        
        // Descripción (AQUÍ está lo importante)
        $data->description = $rawdata['DESCRIPTION'] ??
                            $event->description ??
                            $event->detail ??
                            '';
        
        // Ubicación
        $data->location = $rawdata['LOCATION'] ??
                        $event->location ??
                        '';
        
        // Resto de campos...
        $data->contact = $rawdata['CONTACT'] ?? '';
        $data->uid = $rawdata['UID'] ?? '';
        $data->allDayEvent = $rawdata['allDayEvent'] ?? false;
        $data->geoLat = $rawdata['GEOLAT'] ?? 0;
        $data->geoLon = $rawdata['GEOLON'] ?? 0;
        
        // Fechas
        $data->startDate = $event->startrepeat ?? $event->dtstart ?? '';
        $data->endDate = $event->endrepeat ?? $event->dtend ?? '';
        
        // IDs
        $data->ev_id = $event->ev_id ?? $event->id ?? 0;
        $data->rp_id = $event->rp_id ?? 0;
        $data->catid = $event->catid ?? 0;
        
        $this->debug('Event Data', 'Extracted: ' . ($data->title ?: 'N/A'), 'success', 'list');
        
        return $data;
    }


    /**
     * Obtener datos del evento desde rawdata
     */
    private function parseEventRawdata(?string $rawdata): array
    {
        $data = [
            'DESCRIPTION' => '',
            'SUMMARY' => '',
            'LOCATION' => '',
            'CONTACT' => '',
            'UID' => '',
            'allDayEvent' => false,
            'GEOLON' => 0,
            'GEOLAT' => 0,
        ];
        
        if (empty($rawdata)) {
            return $data;
        }
        
        // Limpiar el string
        $rawdata = trim($rawdata);
        
        // Intentar 1: unserialize normal
        set_error_handler(function() {});
        $unserialized = unserialize($rawdata);
        restore_error_handler();
        
        if (is_array($unserialized)) {
            $this->debug('Rawdata', 'Unserialized successfully', 'success', 'unlock');
            return array_merge($data, $unserialized);
        }
        
        // Intentar 2: Extracción manual
        $fields = [
            'DESCRIPTION' => '/s:11:"DESCRIPTION";s:(\d+):"(.*?)";/s',
            'SUMMARY' => '/s:7:"SUMMARY";s:(\d+):"(.*?)";/s',
            'LOCATION' => '/s:8:"LOCATION";s:(\d+):"(.*?)";/s',
            'CONTACT' => '/s:7:"CONTACT";s:(\d+):"(.*?)";/s',
            'UID' => '/s:3:"UID";s:(\d+):"(.*?)";/s',
        ];
        
        foreach ($fields as $field => $pattern) {
            if (preg_match($pattern, $rawdata, $matches)) {
                $data[$field] = $matches[2] ?? '';
            }
        }
        
        // Extraer valores numéricos
        if (preg_match('/s:6:"GEOLON";d:([\d.]+);/', $rawdata, $matches)) {
            $data['GEOLON'] = (float) $matches[1];
        }
        
        if (preg_match('/s:6:"GEOLAT";d:([\d.]+);/', $rawdata, $matches)) {
            $data['GEOLAT'] = (float) $matches[1];
        }
        
        // Extraer allDayEvent
        if (preg_match('/s:11:"allDayEvent";s:(\d+):"(.*?)";/', $rawdata, $matches)) {
            $data['allDayEvent'] = ($matches[2] === 'on');
        }
        
        return $data;
    }

    /**
     * Helper para debug
     */
    private function debug(string $label, string $message, string $style = 'info', string $icon = 'info'): void
    {
        if ($this->s2sDebugOg) {
            $this->s2sDebugOg->add($label, $message, $style, $icon);
        }
    }
}
