<?php
/**
 * @author          jtotal <support@jtotal.org>
 * @link            https://jtotal.org
 * @copyright       Copyright © 2020 JTOTAL All Rights Reserved
 * @license         GNU GPLv3 <http://www.gnu.org/licenses/gpl.html> or later
 */

defined('_JEXEC') or die;
use Joomla\CMS\Factory;
use Joomla\CMS\Installer\InstallerAdapter;
use Joomla\CMS\Installer\InstallerScriptInterface;
use Joomla\CMS\Language\Text;

return new class implements InstallerScriptInterface
{
    public bool $autopublish = true;
    public $name = 'JT | Social2s';
    public $alias = 'social2s';
    public $extension_type = 'plugin';
    public $plugin_folder = 'content';
    public $minJtfwVersion = '4.0.0';

    public function install(InstallerAdapter $adapter): bool
    {
        //Log::add('Module mod_totalmenu installed correctly.', Log::INFO, 'jerror');
        return true;
    }

    public function update(InstallerAdapter $adapter): bool
    {
        //Log::add('Module mod_totalmenu updated correctly.', Log::INFO, 'jerror');
        return true;
    }

    public function uninstall(InstallerAdapter $adapter): bool
    {
        //Log::add('Module mod_totalmenu uninstalled correctly.', Log::INFO, 'jerror');
        return true;
    }

    public function preflight(string $type, InstallerAdapter $adapter): bool
    {
        $app = Factory::getApplication();

        if (!$this->jtfwInstalled()) {
            $app->enqueueMessage(Text::_('JT framework required. Please, <a target="_blank" href="https://jtotal.org/libs-fw">download at jtotal.org</a> and install it before.'), 'error');
            return false;
        } else {
            $this->checkOldFrameworkUpdate('https://users.jtotal.org/SOFT/UPDATES/PLUGINS/social2s/social2sv4-update.xml');
            $this->checkOldFrameworkUpdate('https://users.jtotal.org/SOFT/UPDATES/PLUGINS/social2s/social2sv5-update.xml');
            $this->checkUpdaterURL(
                'https://users.jtotal.org/SOFT/UPDATES/PLUGINS/social2s/social2s-update.xml'
            );
        }

        return true;
    }

    public function postflight(string $type, InstallerAdapter $adapter): bool
    {
        if ($type === 'install' && $this->autopublish) {
            $this->publishPlugin();
        }

        $app = Factory::getApplication();

        if ($type === 'install') {
            $app->enqueueMessage(
                Text::sprintf('<i class="fa-solid fa-gear"></i> %s has been installed successfully!', $this->name),
                'success'
            );
        } elseif ($type === 'update') {
            $app->enqueueMessage(
                Text::sprintf('<i class="fa-solid fa-gear"></i> %s has been updated successfully!', $this->name),
                'info'
            );
        }

        return true;
    }

    private function jtfwInstalled(): bool
    {
        if (!is_file(JPATH_PLUGINS . '/system/jtframework/autoload.php')) {
            return false;
        }

        $plugin_file = JPATH_PLUGINS . '/system/jtframework/jtframework.xml';
        $xml = $this->parseXMLInstallFile($plugin_file);

        if (!$xml || !isset($xml['version'])) {
            Factory::getApplication()->enqueueMessage("JT framework corrupt. Please, reinstall it.", 'error');
            return false;
        }

        if (version_compare($xml['version'], $this->minJtfwVersion, '<')) {
            Factory::getApplication()->enqueueMessage("JT framework version is too old. Please, update it.", 'error');
            return false;
        }

        return true;
    }
    
    private function parseXMLInstallFile(string $file): array|false
    {
        return \Joomla\CMS\Installer\Installer::parseXMLInstallFile($file);
    }

    private function checkOldFrameworkUpdate(string $siteToDelete): void
    {
        if (!$siteToDelete) {
            return;
        }

        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('COUNT(*) AS count')
            ->from('#__update_sites')
            ->where($db->quoteName('location') . ' = ' . $db->quote($siteToDelete));

        $db->setQuery($query);
        $result = $db->loadResult();

        if ($result > 0) {
            $query = $db->getQuery(true)
                ->delete($db->quoteName('#__update_sites'))
                ->where($db->quoteName('location') . ' = ' . $db->quote($siteToDelete));
            $db->setQuery($query);
            $db->execute();
            Factory::getApplication()->enqueueMessage(Text::_($this->name . ' OLD Update Sites deleted :)'), 'info');
        }
    }

    private function checkUpdaterURL(string $siteToFindAndUpdate): void
    {
        $db = Factory::getDbo();
        
        // Check if site exists
        $query = $db->getQuery(true)
            ->select('COUNT(*) AS count')
            ->from('#__update_sites')
            ->where($db->quoteName('location') . ' = ' . $db->quote($siteToFindAndUpdate));

        $db->setQuery($query);
        $siteExists = $db->loadResult();

        if ($siteExists == 0) {
            // Get extension_id for this plugin
            $query = $db->getQuery(true)
                ->select('extension_id')
                ->from('#__extensions')
                ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'))
                ->where($db->quoteName('element') . ' = ' . $db->quote($this->alias))
                ->where($db->quoteName('folder') . ' = ' . $db->quote($this->plugin_folder));
            $db->setQuery($query);
            $extensionId = $db->loadResult();

            if ($extensionId) {
                // Insert new update site
                $query = $db->getQuery(true)
                    ->insert('#__update_sites')
                    ->columns([$db->quoteName('name'), $db->quoteName('type'), $db->quoteName('location'), $db->quoteName('enabled')])
                    ->values($db->quote($this->name) . ', ' . $db->quote('extension') . ', ' . $db->quote($siteToFindAndUpdate) . ', 1');
                $db->setQuery($query);
                $db->execute();
                
                $updateSiteId = $db->insertid();
                
                // Link update site to extension
                $query = $db->getQuery(true)
                    ->insert('#__update_sites_extensions')
                    ->columns([$db->quoteName('update_site_id'), $db->quoteName('extension_id')])
                    ->values($updateSiteId . ', ' . $extensionId);
                $db->setQuery($query);
                $db->execute();
                
                Factory::getApplication()->enqueueMessage(Text::_($this->name . ' Update Site created :)'), 'info');
            }
        }
    }


    private function publishPlugin(): void
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->update('#__extensions')
            ->set($db->quoteName('enabled') . ' = 1')
            ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'))
            ->where($db->quoteName('element') . ' = ' . $db->quote($this->alias))
            ->where($db->quoteName('folder') . ' = ' . $db->quote($this->plugin_folder));
        $db->setQuery($query);
        $db->execute();
    }

};
