<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  Content.social2s
 *
 * @copyright   Copyright (C) 2005 - 2023 jtotal.org All rights reserved. // Consider updating year
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Jtotal\Plugin\Content\Social2s\Extension;

// Use statements ordered alphabetically for better readability
use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Factory;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Uri\Uri;
use Joomla\Event\DispatcherInterface;
use Joomla\Registry\Registry;
use Jtotal\Plugin\Content\Social2s\Helper\S2sCheckvisibility;
use Jtotal\Plugin\Content\Social2s\Helper\S2sImages;
use Jtotal\Plugin\Content\Social2s\Helper\S2sTmpl;

defined('_JEXEC') or die;

/**
 * Social2s Content Plugin
 *
 * @since  5.0.0
 */
final class Social2s extends CMSPlugin
{
    /**
     * Application object.
     *
     * @var    CMSApplication
     * @since  4.0.0
     */
    protected $app;

    /**
     * Load the language file on instantiation.
     *
     * @var    boolean
     * @since  3.9.0
     */
    protected $autoloadLanguage = true;

    /**
     * Plugin parameters registry object.
     * Must remain public as inherited.
     * @var Registry|null
     */
    public $params = null;

    /**
     * Helper for visibility checks.
     * @var S2sCheckvisibility
     */
    protected $s2sCheck;

    /**
     * Helper for image handling.
     * Accessed internally or via getter if needed.
     * @var S2sImages
     */
    protected $s2sImages; // Changed visibility to protected

    /**
     * Helper for template rendering.
     * @var S2sTmpl
     */
    protected $s2sTmpl;

    /**
     * Helper for OpenGraph tags.
     * @var S2sOg
     */
    protected $s2sOg;

    /**
     * Debug messages storage.
     * @var array<string>
     */
    protected $s2sDebug = [];

    /**
     * OpenGraph debug messages storage.
     * @var array<string>
     */
    protected $s2sOgDebug = [];

    /**
     * Start time for performance measurement.
     * @var float
     */
    protected $startMicro;

    /**
     * Current request view.
     * @var string
     */
    protected $view = '';

    /**
     * Current request component option.
     * @var string
     */
    protected $component = '';

    /**
     * Current request item ID.
     * @var int
     */
    protected $id = 0;

    /**
     * Current request category ID.
     * @var int
     */
    protected $catid = 0;

    /**
     * Current request Itemid (Menu Item ID).
     * @var int
     */
    protected $itemId = 0;

    /**
     * Current processing context (e.g., 'com_content.article').
     * @var string
     */
    protected $context = '';

    /**
     * Style parameters for articles.
     * @var Registry
     */
    protected $artStyleParams;

    /**
     * Style parameters for categories.
     * @var Registry
     */
    protected $catStyleParams;

    /**
     * Icon set to use ('fa', 'img', etc.).
     * @var string
     */
    protected $iconFont = 'fa';

    /**
     * Site name.
     * @var string
     */
    protected $siteName = '';

    /**
     * Base path to the plugin directory URL.
     * @var string
     */
    protected $pluginPath = '';

    /**
     * Base path to the plugin media directory URL.
     * @var string
     */
    protected $mediaPath = '';

    /**
     * Base path to the plugin images directory URL.
     * @var string
     */
    protected $imagesPath = '';

    /**
     * Indicates if the plugin is being rendered within a module context.
     * @var bool
     */
    public $isModule = false; // Public access needed by S2sCheckvisibility::s2sCheckModule

    /**
     * Flag to ensure assets are loaded only once per request.
     * @var bool
     */
    public $assetsLoaded = false; // Added flag for asset loading control

    /**
     * Detected mobile status (true/false). Set by S2sTmpl helper.
     * @var bool|null
     */
    public $isMobileDetected = null;

    /**
     * Detected language tag (e.g., en-GB). Set by S2sTmpl helper.
     * @var string|null
     */
    public $langTag = null;

    /**
     * Detected short language tag (e.g., en). Set by S2sTmpl helper.
     * @var string|null
     */
    public $langTagShort = null;

    /**
     * Detected language direction (ltr/rtl). Set by S2sTmpl helper.
     * @var string|null
     */
    public $langDirection = null;


    /**
     * Constructor.
     *
     * @param   DispatcherInterface  $subject  The object to observe
     * @param   array<string, mixed> $config   An optional associative array of configuration settings.
     */
    public function __construct(DispatcherInterface $subject, array $config = [])
    {
        // Set application object
        $this->app = Factory::getApplication();

        // Set microtime for performance checks early
        $this->startMicro = microtime(true);

        // Ensure parent constructor is called first (this will load params)
        parent::__construct($subject, $config);

        // Load plugin parameters (using $this->params assigned by parent)
        // Fallback check (shouldn't be strictly necessary if parent::__construct works)
        if ($this->params === null) {
             $plugin = PluginHelper::getPlugin('content', 'social2s');
             $this->params = new Registry($plugin->params ?? '');
        }

        // Load language files if not autoloaded (belt and suspenders)
        $this->loadLanguage();

        // Extract request variables
        $jinput = $this->app->input;
        $this->view = $jinput->getString('view', '');
        $this->component = $jinput->getString('option', '');
        $this->id = $jinput->getInt('id', 0);
        $this->catid = $jinput->getInt('catid', 0);
        $this->itemId = $jinput->getInt('Itemid', 0);

        // Initialize derived parameters and configuration
        $this->artStyleParams = new Registry($this->params->get('s2s_display_art', '[]'));
        $this->catStyleParams = new Registry($this->params->get('s2s_display_cat', '[]'));
        $this->iconFont = $this->params->get('s2s_icon_image', 'fa');
        $this->siteName = Factory::getConfig()->get('sitename', '');

        // Calculate paths
        $baseUri = Uri::base();
        $this->pluginPath = $baseUri . 'plugins/content/social2s';
        $this->mediaPath = $baseUri . 'media/plg_content_social2s';
        $this->imagesPath = $this->mediaPath . '/assets/'; // Assuming assets is where images are


        // Instantiate helpers - dependencies of this plugin instance
        $this->s2sCheck = new S2sCheckvisibility($this);
        $this->s2sImages = new S2sImages($this); // Instantiated helper
        $this->s2sTmpl = new S2sTmpl($this);
        //$this->s2sOg = new S2sOg($this);


        // Add initial debug info if needed
        $this->addDebug('Plugin Initialized', get_class($this), 'info', 'cog');
    }


    // --- Getters for properties needed by helpers or layouts ---

    public function getParams(): Registry { return $this->params; }
    public function getView(): string { return $this->view; }
    public function getComponent(): string { return $this->component; }
    public function getId(): int { return $this->id; }
    public function getCatid(): int { return $this->catid; }
    public function getItemId(): int { return $this->itemId; }
    public function getContext(): string { return $this->context; }
    public function getArtStyleParams(): Registry { return $this->artStyleParams; }
    public function getCatStyleParams(): Registry { return $this->catStyleParams; }
    public function getIconFont(): string { return $this->iconFont; }
    public function getSiteName(): string { return $this->siteName; }
    public function getPluginPath(): string { return $this->pluginPath; }
    public function getMediaPath(): string { return $this->mediaPath; }
    public function getImagesPath(): string { return $this->imagesPath; }

    // --- Debug Methods ---

    public function addDebug(string $label, string $message = '', string $style = 'default', string $icon = 'debug', bool $time = true): void
    {
        if (!$this->params->get('s2s_debug', '0')) {
            return;
        }
        $debugHtml = '<td class="text-right"><i class="fa fa-' . htmlspecialchars($icon) . '" aria-hidden="true"></i> <strong>' . htmlspecialchars($label) . ': </strong></td>';
        $debugHtml .= '<td><span class="text-' . htmlspecialchars($style) . '">' . htmlspecialchars($message) . '</span></td>';
        $debugHtml .= '<td class="text-center">';
        if ($time) {
            $currentTime = microtime(true);
            $totalTime = ($currentTime - $this->startMicro) * 1000;
            $debugHtml .= ' <small>' . number_format($totalTime, 2) . ' ms</small>';
        }
        $debugHtml .= '</td>';
        $this->s2sDebug[] = $debugHtml;
    }

    public function addOgDebug(string $label, string $message = '', string $style = 'default', string $icon = 'debug', bool $time = true): void
    {
        if (!$this->params->get('s2s_og_debug', '0')) {
            return;
        }
        $debugHtml = '<td class="text-right"><i class="fa fa-' . htmlspecialchars($icon) . '" aria-hidden="true"></i> <strong>' . htmlspecialchars($label) . ': </strong></td>';
        $debugHtml .= '<td><span class="text-' . htmlspecialchars($style) . '">' . htmlspecialchars($message) . '</span></td>';
        $debugHtml .= '<td class="text-center">';
        if ($time) {
            $currentTime = microtime(true);
            $totalTime = ($currentTime - $this->startMicro) * 1000;
            $debugHtml .= ' <small>' . number_format($totalTime, 2) . ' ms</small>';
        }
        $debugHtml .= '</td>';
        $this->s2sOgDebug[] = $debugHtml;
    }

    protected function renderDebugOutput(): string
    {
        $html = '';
        if ($this->params->get('s2s_debug', '0') && !empty($this->s2sDebug)) {
            $html .= '<div class="table-responsive s2s-debug-output"><table class="table table-striped table-sm table-bordered"><thead><tr class="alert alert-success"><th class="text-center"><i class="fa fa-bug"></i></th><th class="text-center">Social2s Debug</th><th class="text-center"><i class="fa fa-clock-o" aria-hidden="true"></i></th></tr></thead><tbody>';
            foreach ($this->s2sDebug as $value) {
                $html .= '<tr>' . $value . '</tr>';
            }
            $html .= '</tbody></table></div>';
            $this->s2sDebug = [];
        }
        /*
        if ($this->params->get('s2s_og_debug', '0') && !empty($this->s2sOgDebug)) {


            $html .= '<div class="table-responsive s2s-og-debug-output"><table class="table table-striped table-sm table-bordered"><thead><tr class="alert alert-info"><th class="text-center"><i class="fa fa-share-alt"></i></th><th class="text-center">OpenGraph Debug</th><th class="text-center"><i class="fa fa-clock-o" aria-hidden="true"></i></th></tr></thead><tbody>';
            foreach ($this->s2sOgDebug as $value) {
                $html .= '<tr>' . $value . '</tr>';
            }
            $html .= '</tbody></table></div>';
            $this->s2sOgDebug = [];
        }*/
        return $html;
    }

    // --- Core Processing Logic ---

    /**
     * Centralized check to determine if the plugin should process the current event.
     */
    protected function shouldProcess(string $context, $row, $jparams): bool
    {
        $this->context = $context;
        $this->addDebug('Context', $this->context, 'info', 'cube');
        $this->isModule = false;

        // Use helper instance for checks - *** Updated method names ***
        if (!$this->s2sCheck->checkOn()) { // Changed from check_on
            // Debug message added within checkOn
            return false;
        }

        if (!$this->s2sCheck->s2sCheckModule($jparams)) { // Kept s2sCheckModule name
            // Debug message added within s2sCheckModule
            return false;
        }

        if (!$this->s2sCheck->checkContext()) {
            // Debug message added within checkContext
            return false;
        }

        if (!$this->s2sCheck->checkPagesV5($row)) { // Changed from checkPages_v4
            // Debug message added within checkPagesV4
            return false;
        }

        // If all checks passed
        $this->addDebug('Should Process', 'All checks passed', 'success', 'check');
        return true;
    }

    /**
     * Renders the main plugin output (buttons, etc.) and triggers OG tag generation.
     */
    protected function renderPluginOutput($row): string
    {
        // Call the template helper's init method (returns HTML via layout)
        $templateOutput = $this->s2sTmpl->init($row);

        // Call the OpenGraph helper's generateTags method - *** Updated call ***
        // S2sOg needs the $row object and potentially prepared $articleData (passing null for now)
        //MOVIDO A SYSTEM PLUGIN SOCIAL2SOG
        //$this->s2sOg->generateTags($row, null); // Pass null for articleData initially

        return $templateOutput; // Return HTML generated by the template/layout
    }

    // --- Standard Joomla Event Handlers ---

    public function onContentBeforeDisplay(string $context, &$row, &$jparams, ?int $page = 0): ?string
    {
        $this->addDebug('Function', __METHOD__, 'info', 'sign-in');

        if (!$this->shouldProcess($context, $row, $jparams)) {
            return $this->renderDebugOutput();
        }

        // Check position setting - *** Updated method name ***
        $position = (int) $this->s2sCheck->checkPosition(); // Changed from check_position

        if ($position === 0 || $position === 2) { // 0 = Before, 2 = Both
            $this->addDebug('Check Position', 'BEFORE: ' . $position . '. PASSED', 'success', 'toggle-on');
            $output = $this->renderPluginOutput($row); // Generates buttons & OG tags
            return $output . $this->renderDebugOutput(); // Append debug output
        } else {
            $this->addDebug('Check Position', 'BEFORE: ' . $position . '. NOT applicable', 'default', 'toggle-off');
            return $this->renderDebugOutput(); // Render only debug info
        }
    }

    public function onContentAfterDisplay(string $context, &$row, &$jparams, ?int $page = 0): ?string
    {
        $this->addDebug('Function', __METHOD__, 'info', 'sign-out');

        if (!$this->shouldProcess($context, $row, $jparams)) {
            return $this->renderDebugOutput();
        }

        // Check position setting - *** Updated method name ***
        $position = (int) $this->s2sCheck->checkPosition(); // Changed from check_position

        if ($position === 1 || $position === 2) { // 1 = After, 2 = Both
            $this->addDebug('Check Position', 'AFTER: ' . $position . '. PASSED', 'success', 'toggle-on');
            $output = $this->renderPluginOutput($row); // Generates buttons & OG tags
            return $output . $this->renderDebugOutput(); // Append debug output
        } else {
            $this->addDebug('Check Position', 'AFTER: ' . $position . '. NOT applicable', 'default', 'toggle-off');
            // Only render debug if BeforeDisplay didn't already do it
            if ($position !== 0 && $position !== 2) {
                return $this->renderDebugOutput();
            }
            return null; // Position 0 ran, so nothing to add here
        }
    }

    // --- Custom Event Handlers for Integrations ---

    /**
     * Universal handler (e.g., called via {social2s} or other triggers).
     */
    public function ons2sUniversal(string $context, &$row, &$jparams, int $page = 0): ?string // Changed return type hint
    {
         $this->addDebug('Function', __METHOD__, 'info', 'globe');
         $this->context = $context; // Set context manually

        if (!$this->params->get('s2s_display_universal', 0)) {
            $this->addDebug('Universal Display', 'Disabled in params', 'warning', 'ban');
            return $this->renderDebugOutput(); // Return only debug if disabled
        }

        // Perform essential checks (simplified for universal context)
        if (!$this->s2sCheck->checkOn()) {
             $this->addDebug('Check Status', 'Plugin Disabled', 'warning', 'power-off');
            return $this->renderDebugOutput();
        }
        // Potentially skip detailed page/module checks for universal trigger? Depends on requirements.

        // Original check about native support - Refine if necessary
        // if ($this->s2sCheck->checkContext(true)) { // Check for native support
        //     if (!isset($row->s2s_universal_show_on_supported) || !$row->s2s_universal_show_on_supported) {
        //         $this->addDebug('Universal Display', 'Context has native support, skipping Universal.', 'info', 'ban');
        //         return $this->renderDebugOutput();
        //     }
        // }

        $this->addDebug('Universal Display', 'Processing...', 'success', 'check');
        $output = $this->renderPluginOutput($row); // Render main output (includes OG)
        return $output . $this->renderDebugOutput(); // Append debug output
    }

    /**
     * Handler for jEvents.
     */
    public function onJevents($row): ?string // Changed return type hint
    {
     
        $this->addDebug('Function', __METHOD__, 'info', 'calendar');
        $this->context = 'com_jevents.event';

        if (!$this->params->get('s2s_jevents', 0)) {
            $this->addDebug('jEvents Display', 'Disabled in params', 'warning', 'ban');
            return $this->renderDebugOutput();
        }

        // Simplified checks for integration handler
        if (!$this->s2sCheck->checkOn()) {
            $this->addDebug('Check Status', 'Plugin Disabled', 'warning', 'power-off');
            return $this->renderDebugOutput();
        }
        // Add specific jEvents item filtering/checks here if needed
        $this->addDebug('jEvents Display', 'Processing...', 'success', 'check');
        $output = $this->renderPluginOutput($row);
        return $output . $this->renderDebugOutput();
    }

     /**
     * Handler for SP Page Builder / SP Soccer.
     */
    public function onSP(string $context, &$row, &$params, int $page = 0): ?string
    {
        $this->addDebug('Function', __METHOD__, 'info', 'puzzle-piece');
        $this->context = $context;

        // Allow specific SP contexts
        if ($context !== 'com_sppagebuilder.page' && $context !== 'com_spsoccer.page') {
            $this->addDebug('SP Component', 'Context not supported: ' . $context, 'warning', 'ban');
            return $this->renderDebugOutput();
        }

        // Simplified checks for integration handler
        if (!$this->s2sCheck->checkOn()) {
             $this->addDebug('Check Status', 'Plugin Disabled', 'warning', 'power-off');
             return $this->renderDebugOutput();
        }
         // Add specific SP item filtering/checks here if needed

        $this->addDebug('SP Component', 'Processing context: ' . $context, 'success', 'check');
        $output = $this->renderPluginOutput($row); // Render main output (includes OG)
        return $output . $this->renderDebugOutput(); // Append debug output
    }

    // --- Static Methods ---

    /**
     * Static function to get documentation link/info.
     */
    public static function getDoc(): string
    {
        // Example: return 'https://docs.jtotal.org/social2s';
        return ''; // Placeholder
    }


    public function getS2sImagesHelper(): S2sImages
    {
        return $this->s2sImages;
    }

    


}