<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  Content.social2s
 *
 * @copyright   Copyright (C) 2005 - 2022 jtotal.org All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Jtotal\Plugin\Content\Social2s\Helper;

use Joomla\Registry\Registry;
use Jtotal\Plugin\Content\Social2s\Extension\Social2s; // Keep for type hinting

defined('_JEXEC') or die;

/**
 * Helper class for visibility checks within the Social2s plugin.
 */
class S2sCheckvisibility
{
    /**
     * The main plugin instance.
     * Provides access to parameters, configuration, and context.
     *
     * @var Social2s
     */
    protected $plugin;

    /**
     * Constructor.
     *
     * @param Social2s $plugin The main Social2s plugin instance.
     */
    public function __construct(Social2s $plugin)
    {
        $this->plugin = $plugin;
    }

    /**
     * Checks if the plugin is generally enabled for the current view.
     *
     * @return bool True if enabled for the view, false otherwise.
     */
    public function checkOn(): bool
    {
        $params = $this->plugin->getParams();
        $view = $this->plugin->getView(); // Get view from plugin instance

        // If operating within a module context, adjust the view check logic if needed
        // Example: maybe modules always use 'category' settings?
        // if ($this->plugin->isModule) {
        //    $view = 'category'; // Or apply specific module logic
        // }

        $art_on = $params->get('s2s_display_art_view', '1') === '1';
        $cat_on = $params->get('s2s_display_cat_view', '1') === '1';
        $feat_on = $params->get('s2s_featured_inc_exc', '1') === '1'; // Assuming 1 means enabled
        $s2s_takefromarticle = $params->get('s2s_takefromarticle', '1') === '1';

        if ($s2s_takefromarticle) {
            // If category settings are derived from article settings
            $cat_on = $art_on;
            $feat_on = $art_on; // Featured likely uses category/article settings too
        }

        $check_view = true; // Assume enabled unless proven otherwise

        // Check based on the actual view
        switch ($view) {
            case 'article':
                if (!$art_on) {
                    $this->plugin->addDebug('Check Activation', $view . ' view disabled', 'warning', 'toggle-off');
                    $check_view = false;
                }
                break;

            case 'category':
                if (!$cat_on) {
                    $this->plugin->addDebug('Check Activation', $view . ' view disabled', 'warning', 'toggle-off');
                    $check_view = false;
                }
                break;

            case 'featured':
                // Featured usually involves articles, check both featured setting and underlying type setting
                if (!$feat_on || !$art_on) { // Or perhaps check !$cat_on depending on config
                    $this->plugin->addDebug('Check Activation', $view . ' view disabled', 'warning', 'toggle-off');
                    $check_view = false;
                }
                break;

                // Add cases for other supported views like VirtueMart if necessary
            case 'productdetails': // Assuming com_virtuemart view is 'productdetails'
                // Check specific VirtueMart setting if it exists
                $vm_on = $params->get('s2s_display_vm_view', '1') === '1'; // Example param name
                if (!$vm_on) {
                     $this->plugin->addDebug('Check Activation', 'VirtueMart view disabled', 'warning', 'toggle-off');
                     $check_view = false;
                }
                break;

            default:
                // If the view is not explicitly handled, assume it's disabled unless a default allows it
                // $this->plugin->addDebug('Check Activation', $view . ' view not configured for activation check', 'info', 'question-circle');
                // Decide if unhandled views should be allowed or not. Maybe return true if plugin should attempt?
                // For safety, let's assume unhandled standard views are disabled unless part of a specific integration check
                // $check_view = false;
                break;
        }

        if ($check_view) {
             $this->plugin->addDebug('Check Activation', 'View: ' . $view . ' - Passed', 'success', 'toggle-on');
        }

        return $check_view;
    }

    /**
     * Determines the configured display position (before/after/both).
     *
     * @return int 0 for Before, 1 for After, 2 for Both.
     */
    public function checkPosition(): int
    {
        $context = $this->plugin->getContext();
        $params = $this->plugin->getParams();

        // Default position usually comes from article settings
        $artStyleParams = $this->plugin->getArtStyleParams();
        $defaultPos = $artStyleParams->get('s2s_pos', 1); // Default to 1 (After) if not set

        $s2s_pos = (int) $defaultPos; // Start with default

        // Check if we are in a category/featured view context
        if ($context === 'com_content.category' || $context === 'com_content.featured') {
            $s2s_takefromarticle = $params->get('s2s_takefromarticle', 1) == 1; // Use '1' for boolean check

            if ($s2s_takefromarticle) {
                // Use the article position setting already determined
                $s2s_pos = (int) $artStyleParams->get('s2s_pos', $defaultPos);
                 $this->plugin->addDebug('Position Source', 'Using Article Setting ('.$s2s_pos.') for '.$context, 'info', 'link');
            } else {
                // Use the specific category position setting
                $catStyleParams = $this->plugin->getCatStyleParams();
                $s2s_pos = (int) $catStyleParams->get('s2s_pos', $defaultPos); // Fallback to default if cat setting missing
                $this->plugin->addDebug('Position Source', 'Using Category Setting ('.$s2s_pos.') for '.$context, 'info', 'link');
            }
        } elseif ($context === 'com_virtuemart.productdetails') {
             // Example: Check for a VM specific position setting
             $vmStyleParams = new Registry($params->get('s2s_display_vm', '[]')); // Assuming VM settings are here
             $s2s_pos = (int) $vmStyleParams->get('s2s_pos', $defaultPos); // Fallback to default
             $this->plugin->addDebug('Position Source', 'Using VirtueMart Setting ('.$s2s_pos.')', 'info', 'link');
        } else {
            // For articles or other contexts, the default (article) position applies
             $s2s_pos = (int) $artStyleParams->get('s2s_pos', $defaultPos);
             $this->plugin->addDebug('Position Source', 'Using Article/Default Setting ('.$s2s_pos.') for '.$context, 'info', 'link');
        }


        return $s2s_pos;
    }

    /**
     * Checks if the plugin is being rendered within a module context
     * and if modules are allowed. Sets the plugin's isModule flag.
     *
     * @param mixed $jparams Parameters passed to the event handler (often a Registry or array).
     *
     * @return bool True if processing should continue (module allowed or not a module), false if it's a disallowed module.
     */
    public function s2sCheckModule($jparams): bool
    {
        // Reset flag at the start of the check
        $this->plugin->isModule = false;

        $s2s_display_module = $this->plugin->getParams()->get('s2s_display_module', 0);

        // Try to detect if it's a module based on parameters structure
        // This detection logic might need refinement based on how Joomla modules pass params
        $isLikelyModule = false;
        if ($jparams instanceof Registry) {
            // Common pattern: modules might have specific keys like 'moduleclass_sfx' or structure
            // The original check used ->get('ordering'), which seems unreliable.
            // Let's check for a common module parameter like 'moduleclass_sfx' or 'module_tag'
            if ($jparams->get('moduleclass_sfx') !== null || $jparams->get('module_tag') !== null) {
                $isLikelyModule = true;
            }
            // Alternatively, check if the params look like module params registry
            // e.g., $jparams->get('params') instanceof Registry
        } elseif (is_array($jparams) && (isset($jparams['moduleclass_sfx']) || isset($jparams['module_tag']))) {
             $isLikelyModule = true;
        }


        if ($isLikelyModule) {
            $this->plugin->isModule = true; // Set the flag on the plugin instance
            $this->plugin->addDebug('Context Type', 'Module Detected', 'info', 'puzzle-piece');

            if ($s2s_display_module == 0) {
                $this->plugin->addDebug('Check Module Display', 'Modules are disabled - Skipping', 'warning', 'ban');
                return false; // Do not continue if modules are disabled
            } else {
                $this->plugin->addDebug('Check Module Display', 'Modules are enabled - Proceeding', 'success', 'check');
                // DO NOT change $this->plugin->view here. Let other checks use the module flag.
                return true; // Continue processing for allowed modules
            }
        } else {
             $this->plugin->addDebug('Context Type', 'Not a Module (or not detected)', 'info', 'file-text-o');
            // Not a module (or detection failed), continue normal processing
            return true;
        }
    }

    /**
     * Checks if the current context (component.view) is supported by the plugin.
     *
     * @param bool $universal Set to true if checking in a universal context (e.g., from a shortcode trigger).
     *                        In universal mode, it might just log differently if context isn't natively supported.
     *
     * @return bool True if the context is supported or if in universal mode, false otherwise.
     */
    public function checkContext(bool $universal = false): bool
    {
        $context = $this->plugin->getContext();
        $params = $this->plugin->getParams();

        // Define supported contexts based on plugin settings
        $supportedContexts = [];
        if ($params->get('s2s_display_art_view', '1') === '1' || $params->get('s2s_takefromarticle', '1') === '1') {
            $supportedContexts[] = 'com_content.article';
        }
        if ($params->get('s2s_display_cat_view', '1') === '1' || $params->get('s2s_takefromarticle', '1') === '1') {
             $supportedContexts[] = 'com_content.category';
             // Featured view often uses category/article settings
            if ($params->get('s2s_featured_inc_exc', '1') === '1') {
                 $supportedContexts[] = 'com_content.featured';
            }
        }
         // Example: Check if VirtueMart integration is enabled
        if ($params->get('s2s_vm_enable', '0') === '1' && $params->get('s2s_display_vm_view', '1') === '1') { // Check specific VM enable param
            $supportedContexts[] = 'com_virtuemart.productdetails';
             // Add other VM contexts if supported: 'com_virtuemart.category', etc.
        }
        // Add other integrations (jEvents, SP PB etc.) based on their enable flags
        if ($params->get('s2s_jevents', 0) === '1') {
             // Add relevant jEvents contexts, e.g., 'com_jevents.event' - Adjust as needed
             // $supportedContexts[] = 'com_jevents.event';
        }
        // SP Page Builder / SP Soccer (Contexts checked in onSP directly in original code)
        // We can add them here for consistency if desired
        // $supportedContexts[] = 'com_sppagebuilder.page';
        // $supportedContexts[] = 'com_spsoccer.page';


        // Check if the current context is in the list of supported contexts
        if (in_array($context, $supportedContexts)) {
            $this->plugin->addDebug('Check Context Support', $context . ' - Supported', 'success', 'check-circle');
            return true;
        } else {
            // Context is not directly supported
            if ($universal) {
                // In universal mode, we might still proceed but log it differently
                $this->plugin->addDebug('Check Context Support', $context . ' - Not natively supported (Universal Mode)', 'info', 'info-circle');
                // Decide if universal mode should *always* return true here,
                // or if it depends on whether the context *could* potentially work.
                // Returning true allows the universal handler to decide further.
                return true;
            } else {
                // In normal event handling, unsupported contexts should stop processing.
                $this->plugin->addDebug('Check Context Support', $context . ' - Not Supported', 'danger', 'times-circle');
                return false;
            }
        }
    }

    /**
     * Checks if the current article/category is included/excluded based on plugin settings.
     * Uses parameters directly from the plugin instance.
     *
     * @param object $row The content object (article, category, product etc.), expected to have 'id' and 'catid'.
     *
     * @return bool True if the item should be displayed, false otherwise.
     */
    public function checkPagesV5($row): bool
    {
        $params = $this->plugin->getParams();
        $context = $this->plugin->getContext();

        // --- Contexts with Free Pass (if any) ---
        // Example: Allow VirtueMart product details regardless of category/article ID filters
        if ($context === 'com_virtuemart.productdetails') {
             // Check if VM has its own include/exclude rules, otherwise grant pass
            $vm_filter_enabled = $params->get('vm_filter_enable', 0); // Example param
            if (!$vm_filter_enabled) {
                $this->plugin->addDebug('Check Item Filtering', 'VirtueMart context bypasses filters', 'info', 'forward');
                return true;
            }
             // If VM filtering is enabled, continue to checks below using relevant VM IDs
        }
        // Add other contexts that might bypass standard filtering


        // --- Standard Article/Category Filtering ---
        $category_id = $row->catid ?? null; // Use null coalescing for safety
        $article_id = $row->id ?? null;

        if ($category_id === null && $context !== 'com_content.article') {
            // Category ID is crucial for category/featured views, but might be missing in pure article view
             // If it's an article view without catid, maybe filtering doesn't apply or uses only article ID?
             // Let's allow article view to proceed to article ID check
             // For other views, missing catid is usually an issue
            if ($context !== 'com_content.article') {
                 $this->plugin->addDebug('Check Item Filtering', 'Missing Category ID for context: '.$context, 'danger', 'exclamation-triangle');
                 return false;
            } else {
                $this->plugin->addDebug('Check Item Filtering', 'Missing Category ID for Article - Proceeding with Article ID check', 'info', 'exclamation-triangle');
            }
        }
        if ($article_id === null && $context === 'com_content.article') {
             $this->plugin->addDebug('Check Item Filtering', 'Missing Article ID for context: '.$context, 'danger', 'exclamation-triangle');
             return false; // Article context needs article ID
        }


        // Get category filter settings
        $catidsSetting = $params->get('categories', '');
        // Ensure catids is an array, handle legacy string/empty cases
        $catids = is_array($catidsSetting) ? $catidsSetting : ($catidsSetting ? explode(',', $catidsSetting) : []);
        $catids = array_filter(array_map('trim', $catids), 'strlen'); // Clean up array
        $cat_exc_inc = $params->get('cat_inc_exc', '1'); // 1 = Include, 0 = Exclude

        // Get article filter settings
        $articleIdsSetting = $params->get('article_ids', '');
         // Ensure article_ids is an array
        $article_ids = is_array($articleIdsSetting) ? $articleIdsSetting : ($articleIdsSetting ? explode(',', $articleIdsSetting) : []);
        $article_ids = array_filter(array_map('trim', $article_ids), 'strlen'); // Clean up array
        $article_exc_inc = $params->get('article_inc_exc', '1'); // 1 = Include, 0 = Exclude


        // --- Filtering Logic ---
        $categoryAllowed = true; // Assume allowed by category unless excluded
        $articleAllowed = true; // Assume allowed by article unless excluded

        // ** Category Check **
        if (!empty($catids) && $category_id !== null) { // Only filter if categories are specified and we have a catid
            $isInCategoryList = in_array((string)$category_id, $catids); // Ensure comparison with strings

            if ($cat_exc_inc === '1') { // Mode: Include selected categories
                $categoryAllowed = $isInCategoryList;
                $modeStr = 'Include List';
                $this->plugin->addDebug('Category Filter ('.$modeStr.')', 'CatID: '.$category_id . ($categoryAllowed ? ' - Matched' : ' - Not Matched'), $categoryAllowed ? 'info' : 'warning', 'filter');
            } else { // Mode: Exclude selected categories
                $categoryAllowed = !$isInCategoryList;
                 $modeStr = 'Exclude List';
                $this->plugin->addDebug('Category Filter ('.$modeStr.')', 'CatID: '.$category_id . ($categoryAllowed ? ' - Not Excluded' : ' - Excluded'), $categoryAllowed ? 'info' : 'warning', 'filter');
            }
        } else {
             $this->plugin->addDebug('Category Filter', 'No specific category filters applied', 'info', 'filter');
        }


        // ** Article Check ** (Only if category check passed, unless article filter overrides)
        // Decide precedence: Does article exclusion override category inclusion? Usually yes.
        // Does article inclusion override category exclusion? Often yes for specific articles.

        if (!empty($article_ids) && $article_id !== null) { // Only filter if articles are specified and we have an article id
            $isInArticleList = in_array((string)$article_id, $article_ids);

            if ($article_exc_inc === '1') { // Mode: Include selected articles
                $articleAllowed = $isInArticleList;
                $modeStr = 'Include List';
                 $this->plugin->addDebug('Article Filter ('.$modeStr.')', 'ArtID: '.$article_id . ($articleAllowed ? ' - Matched' : ' - Not Matched'), $articleAllowed ? 'info' : 'warning', 'filter');
                // Precedence: If specific articles are INCLUDED, they should show even if category was excluded.
                if ($articleAllowed) return true; // Force allow if article is explicitly included

            } else { // Mode: Exclude selected articles
                $articleAllowed = !$isInArticleList;
                $modeStr = 'Exclude List';
                $this->plugin->addDebug('Article Filter ('.$modeStr.')', 'ArtID: '.$article_id . ($articleAllowed ? ' - Not Excluded' : ' - Excluded'), $articleAllowed ? 'info' : 'warning', 'filter');
                 // Precedence: If specific articles are EXCLUDED, they should NOT show even if category was included.
                 if (!$articleAllowed) return false; // Force disallow if article is explicitly excluded
            }
        } else {
              $this->plugin->addDebug('Article Filter', 'No specific article filters applied', 'info', 'filter');
        }


        // Final decision based on category filtering if no article override occurred
        $finalDecision = $categoryAllowed; // Default to category result

        if ($finalDecision) {
            $this->plugin->addDebug('Check Item Filtering', 'PASSED (CatID: '.$category_id.', ArtID: '.$article_id.')', 'success', 'check');
        } else {
            $this->plugin->addDebug('Check Item Filtering', 'NOT PASSED (CatID: '.$category_id.', ArtID: '.$article_id.')', 'danger', 'ban');
        }

        return $finalDecision;
    }


     /**
     * [DEPRECATED/UNUSED?] Original checkComponent function.
     * Its logic seems incorporated into checkContext or specific event handlers now.
     * Kept for reference, but likely safe to remove if not called elsewhere.
     *
     * @param string $component The component name (e.g., 'com_content').
     * @param string $view      The view name (e.g., 'article').
     * @return bool
     */
    public function checkComponent(string $component, string $view): bool
    {
        $this->plugin->addDebug('Legacy Check', 'checkComponent called (likely unused)', 'warning', 'question');
        // Original logic:
        if ($component == 'com_content') {
            if ($view == 'article' || $view == 'category' || $view == 'featured') {
                return true;
            }
        } elseif ($component == 'com_virtuemart') {
            // Consider checking specific VM views if needed
            return true;
        } else {
            $this->plugin->addDebug('Legacy Check Component', $component . ' not supported', 'danger', 'toggle-on');
            return false;
        }
        return false; // Default fallback
    }

     /**
      * [DEPRECATED/UNUSED?] Original checkComponentSupport function.
      * Its logic seems unclear and it always returned true.
      * Kept for reference, but likely safe to remove.
      *
      * @return bool Always true in original implementation.
      */
    public function checkComponentSupport(): bool
    {
         $this->plugin->addDebug('Legacy Check', 'checkComponentSupport called (likely unused)', 'warning', 'question');
        // $context = $this->plugin->getContext();
        // $params = $this->plugin->getParams();
        // Original logic simply returned true.
        return true;
    }
}