<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  Content.social2s
 *
 * @copyright   Copyright (C) 2005 - 2022 jtotal.org All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Jtotal\Plugin\Content\Social2s\Helper;

use DOMDocument;
use Joomla\Filesystem\File;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Uri\Uri; // Added for base URL construction
use Jtotal\Plugin\Content\Social2s\Extension\Social2s; // Keep for type hinting

defined('_JEXEC') or die;

/**
 * Helper class for extracting and preparing images for Social2s plugin.
 */
class S2sImages
{
    /**
     * The main plugin instance.
     * Provides access to parameters, configuration, and context.
     *
     * @var Social2s
     */
    protected $plugin;

    /**
     * Constructor.
     *
     * @param Social2s $plugin The main Social2s plugin instance.
     */
    public function __construct(Social2s $plugin)
    {
        $this->plugin = $plugin;
    }

    /**
     * Get images from a standard Joomla com_content article object.
     *
     * @param object $article The Joomla article object.
     *
     * @return array<string> An array of prepared image URLs.
     */
    public function getImagesArticle(object $article): array
    {

      
        $params = $this->plugin->getParams();
        $og_add_dom_img = $params->get('og_add_dom_img', '0'); // 0=No, 1=First, 2=All

        // Check if 'Always Default' image setting is active
        $alwaysImages = $this->prepareDefAlways();
        if ($alwaysImages !== false) {
            return $alwaysImages; // Return immediately if default image is set to always override
        }

        $candidateImages = [];

        // --- Extract images from Article properties ---
        if (!empty($article->images)) {
            // Decode JSON image string safely
            try {
                $introimages = json_decode($article->images, false, 512, JSON_THROW_ON_ERROR);

                // Fulltext image
                if (!empty($introimages->image_fulltext)) {
                    $this->plugin->addOgDebug('Found Fulltext Img', $introimages->image_fulltext, 'info', 'image');
                    $candidateImages[] = $introimages->image_fulltext;
                }

                // Intro image (check if skipped)
                if (!$params->get('og_skip_intro_img', '0')) {
                    if (!empty($introimages->image_intro)) {
                        $this->plugin->addOgDebug('Found Intro Img', $introimages->image_intro, 'info', 'image');
                        $candidateImages[] = $introimages->image_intro;
                    }
                }
            } catch (\JsonException $e) {
                  $this->plugin->addOgDebug('Image JSON Error', $e->getMessage(), 'danger', 'exclamation-triangle');
            }
        }

        // --- Handle Single Image Mode Early ---
        // If only one image is needed and we already found one from article properties
        if ($params->get('og_multi_img', '0') == '0') {
            $preparedSingle = $this->prepareImages($candidateImages, 1); // Prepare only the first found image
            if (!empty($preparedSingle)) {
                 $this->plugin->addOgDebug('Using Single Image', $preparedSingle[0], 'success', 'image');
                return $preparedSingle; // Return the single prepared image
            }
            // If no image found yet, continue to DOM check (if enabled)
        }

        // --- Extract images from DOM ---
        if ($og_add_dom_img !== '0' && isset($article->introtext)) {
            $findAllInDom = ($og_add_dom_img === '2'); // Find all if setting is '2'
            $dom_images = $this->getDomImage($article->introtext, $findAllInDom);
            if (!empty($dom_images)) {
                $candidateImages = array_merge($candidateImages, $dom_images);
                $this->plugin->addOgDebug('Found DOM Images', count($dom_images) . ' image(s)', 'info', 'code');
            }
        }

        // --- Prepare all collected images ---
        $preparedImages = $this->prepareImages($candidateImages);

        // --- Apply Default Image as Last Resort ---
        if (empty($preparedImages)) {
            // Check if default image should be used as fallback (option '3')
            if ($params->get('opengraph_default_image_opt', '0') == '3') {
                $defaultImg = $params->get('opengraph_default_image', '');
                if (!empty($defaultImg)) {
                    // Prepare the default image (cleans URL, but size check might be skipped for default)
                     $preparedDefault = $this->prepareImages([$defaultImg], 1, true); // Skip size check for default? Add flag
                    if (!empty($preparedDefault)) {
                        $preparedImages = $preparedDefault;
                        $this->plugin->addOgDebug('Using Default Image', '(Last Resort) ' . $preparedImages[0], 'info', 'image');
                    }
                }
            }
        }

        $this->plugin->addOgDebug('Final Article Images', !empty($preparedImages) ? count($preparedImages) . ' image(s) selected' : 'None', 'success', 'image');
        return $preparedImages;
    }

    /**
     * Get images from a jEvents event object.
     * Assumes event description is in $event->_description.
     *
     * @param object $event The jEvents event object.
     *
     * @return array<string> An array of prepared image URLs.
     */
    public function getImagesJevents(object $event): array
    {
        $params = $this->plugin->getParams();
        $og_add_dom_img = $params->get('og_add_dom_img', '0'); // 0=No, 1=First, 2=All

        // Check if 'Always Default' image setting is active
        $alwaysImages = $this->prepareDefAlways();
        if ($alwaysImages !== false) {
            return $alwaysImages;
        }

        $candidateImages = [];

        // jEvents typically doesn't have structured image fields like com_content.
        // Rely primarily on DOM parsing of the description.
        if ($og_add_dom_img !== '0' && isset($event->_description)) {
            $findAllInDom = ($og_add_dom_img === '2');
            $dom_images = $this->getDomImage($event->_description, $findAllInDom);
            if (!empty($dom_images)) {
                $candidateImages = array_merge($candidateImages, $dom_images);
                $this->plugin->addOgDebug('Found jEvents DOM Images', count($dom_images) . ' image(s)', 'info', 'code');
            }
        }

        // --- Prepare collected images ---
        $preparedImages = $this->prepareImages($candidateImages);

        // --- Apply Default Image as Last Resort ---
        if (empty($preparedImages) && $params->get('opengraph_default_image_opt', '0') == '3') {
             $defaultImg = $params->get('opengraph_default_image', '');
            if (!empty($defaultImg)) {
                $preparedDefault = $this->prepareImages([$defaultImg], 1, true); // Skip size check for default
                if (!empty($preparedDefault)) {
                    $preparedImages = $preparedDefault;
                    $this->plugin->addOgDebug('Using Default Image', '(Last Resort) ' . $preparedImages[0], 'info', 'image');
                }
            }
        }

        $this->plugin->addOgDebug('Final jEvents Images', !empty($preparedImages) ? count($preparedImages) . ' image(s) selected' : 'None', 'success', 'image');
        return $preparedImages;
    }

    /**
     * Get images from a K2 item object.
     *
     * @param object $item The K2 item object.
     *
     * @return array<string> An array of prepared image URLs.
     */
    public function getImagesK2(object $item): array
    {
        $params = $this->plugin->getParams();
        $og_add_dom_img = $params->get('og_add_dom_img', '0'); // 0=No, 1=First, 2=All

        // Check if 'Always Default' image setting is active
        $alwaysImages = $this->prepareDefAlways();
        if ($alwaysImages !== false) {
            return $alwaysImages;
        }

        $candidateImages = [];

        // --- Extract K2 image ---
        // K2 stores images based on item ID hash
        $imageIdHash = md5("Image" . $item->id);
        $k2ImagePathPrefix = 'media/k2/items/cache/' . $imageIdHash;

        // Check for standard K2 image sizes (XL first)
        if (File::exists(JPATH_SITE . '/' . $k2ImagePathPrefix . '_XL.jpg')) {
            $candidateImages[] = $k2ImagePathPrefix . '_XL.jpg';
             $this->plugin->addOgDebug('Found K2 Image', $candidateImages[0], 'info', 'image');
        } elseif (File::exists(JPATH_SITE . '/' . $k2ImagePathPrefix . '_Generic.jpg')) {
             $candidateImages[] = $k2ImagePathPrefix . '_Generic.jpg';
             $this->plugin->addOgDebug('Found K2 Image', '(Generic) ' . $candidateImages[0], 'info', 'image');
        }
        // Add checks for other sizes (_L, _M, _S, _XS) if needed, based on priority

        // --- Handle Single Image Mode Early ---
        if ($params->get('og_multi_img', '0') == '0') {
            $preparedSingle = $this->prepareImages($candidateImages, 1);
            if (!empty($preparedSingle)) {
                $this->plugin->addOgDebug('Using Single K2 Image', $preparedSingle[0], 'success', 'image');
                return $preparedSingle;
            }
        }

        // --- Extract images from DOM ---
        if ($og_add_dom_img !== '0' && isset($item->introtext)) { // K2 uses introtext too
            $findAllInDom = ($og_add_dom_img === '2');
            $dom_images = $this->getDomImage($item->introtext, $findAllInDom);
            if (!empty($dom_images)) {
                $candidateImages = array_merge($candidateImages, $dom_images);
                $this->plugin->addOgDebug('Found K2 DOM Images', count($dom_images) . ' image(s)', 'info', 'code');
            }
        }

        // --- Prepare all collected images ---
        $preparedImages = $this->prepareImages($candidateImages);

        // --- Apply Default Image as Last Resort ---
        if (empty($preparedImages) && $params->get('opengraph_default_image_opt', '0') == '3') {
            $defaultImg = $params->get('opengraph_default_image', '');
            if (!empty($defaultImg)) {
                $preparedDefault = $this->prepareImages([$defaultImg], 1, true);
                if (!empty($preparedDefault)) {
                    $preparedImages = $preparedDefault;
                    $this->plugin->addOgDebug('Using Default Image', '(Last Resort) ' . $preparedImages[0], 'info', 'image');
                }
            }
        }

        $this->plugin->addOgDebug('Final K2 Images', !empty($preparedImages) ? count($preparedImages) . ' image(s) selected' : 'None', 'success', 'image');
        return $preparedImages;
    }

    /**
     * Get images from a VirtueMart product object.
     * Assumes product object has an 'images' property (array of image objects)
     * and 'product_desc' for DOM parsing.
     *
     * @param object $product The VirtueMart product object.
     *
     * @return array<string> An array of prepared image URLs.
     */
    public function getImagesVirtuemart(object $product): array
    {
        $params = $this->plugin->getParams();
        $og_add_dom_img = $params->get('og_add_dom_img', '0'); // 0=No, 1=First, 2=All

        // Check if 'Always Default' image setting is active
        $alwaysImages = $this->prepareDefAlways();
        if ($alwaysImages !== false) {
            return $alwaysImages;
        }

        // Determine max images based on single/multi setting
        $maxImages = ($params->get('og_multi_img', '0') == '0')
            ? 1
            : (int) $params->get('og_max_multi_img', 3); // Ensure max is integer

        $candidateImages = [];

        // --- Extract from VirtueMart image objects ---
        if (!empty($product->images) && is_array($product->images)) {
            foreach ($product->images as $image) {
                // Check if we already have enough images
                if (count($candidateImages) >= $maxImages) {
                    break;
                }

                // Basic check if it looks like a valid image object and URL
                // VM specific checks: file_is_forSale == '0', file_mimetype != ''
                if (isset($image->file_url) && !empty($image->file_url) &&
                    isset($image->file_is_forSale) && $image->file_is_forSale == '0' &&
                    isset($image->file_mimetype) && $image->file_mimetype != '') {
                    // URL might already be absolute in VM, no need for utf8_decode generally
                    $candidateImages[] = $image->file_url;
                    $this->plugin->addOgDebug('Found VM Image Object', $image->file_url, 'info', 'image');
                }
            }
        }

        // --- Handle Single Image Mode Early ---
        if ($params->get('og_multi_img', '0') == '0') {
            $preparedSingle = $this->prepareImages($candidateImages, 1);
            if (!empty($preparedSingle)) {
                $this->plugin->addOgDebug('Using Single VM Image', $preparedSingle[0], 'success', 'image');
                return $preparedSingle;
            }
        }

        // --- Extract images from DOM (if needed and enabled) ---
        if (count($candidateImages) < $maxImages && $og_add_dom_img !== '0' && isset($product->product_desc)) {
            $findAllInDom = ($og_add_dom_img === '2');
            $dom_images = $this->getDomImage($product->product_desc, $findAllInDom);
            if (!empty($dom_images)) {
                // Add DOM images until max is reached
                $needed = $maxImages - count($candidateImages);
                if ($needed > 0) {
                    $imagesToAdd = array_slice($dom_images, 0, $needed);
                    $candidateImages = array_merge($candidateImages, $imagesToAdd);
                    $this->plugin->addOgDebug('Found VM DOM Images', count($imagesToAdd) . ' image(s)', 'info', 'code');
                }
            }
        }

        // --- Prepare all collected images ---
        // Pass the already determined $maxImages limit to prepareImages
        $preparedImages = $this->prepareImages($candidateImages, $maxImages);

        // --- Apply Default Image as Last Resort ---
        if (empty($preparedImages) && $params->get('opengraph_default_image_opt', '0') == '3') {
            $defaultImg = $params->get('opengraph_default_image', '');
            if (!empty($defaultImg)) {
                $preparedDefault = $this->prepareImages([$defaultImg], 1, true);
                if (!empty($preparedDefault)) {
                    $preparedImages = $preparedDefault;
                    $this->plugin->addOgDebug('Using Default Image', '(Last Resort) ' . $preparedImages[0], 'info', 'image');
                }
            }
        }

        $this->plugin->addOgDebug('Final VM Images', !empty($preparedImages) ? count($preparedImages) . ' image(s) selected' : 'None', 'success', 'image');
        return $preparedImages;
    }

    /**
     * Extracts image URLs from HTML content using DOM parsing.
     *
     * @param string $html       The HTML content to parse.
     * @param bool   $extractAll If true, extracts all found images; otherwise, stops after the first valid one.
     *
     * @return array<string> An array of found image URLs.
     */
    private function getDomImage(string $html, bool $extractAll = false): array
    {
        $images = [];
        if (empty(trim($html))) {
            return $images; // Return empty if HTML is empty
        }

        $domdoc = new DOMDocument();
        // Use libxml to suppress errors during loading of potentially invalid HTML
        libxml_use_internal_errors(true);
        $domdoc->loadHTML($html);
        libxml_clear_errors(); // Clear any reported errors

        $tags = $domdoc->getElementsByTagName('img');

        foreach ($tags as $tag) {
            if ($tag->hasAttribute('src')) {
                // Decode potentially HTML entities in src, use URL from source directly
                $imageUrl = $tag->getAttribute('src');

                // Basic check for empty src or data URIs if they should be skipped
                if (empty($imageUrl) || strpos($imageUrl, 'data:image') === 0) {
                    continue;
                }

                // Optionally check image size here, or defer to prepareImages
                 // For performance, better to defer size check to prepareImages if possible
                // if ($this->imgSize($imageUrl)) {
                $images[] = $imageUrl;
                if (!$extractAll) {
                    break; // Stop after the first one if $extractAll is false
                }
                // }
            }
        }

        return $images;
    }

/**
     * Checks if an image meets the minimum size requirements specified in plugin parameters.
     * Assumes $imgUrl is already an absolute URL.
     *
     * @param string $imgUrl The absolute URL of the image to check.
     *
     * @return bool True if the image meets size requirements or if checking is disabled, false otherwise.
     */
    private function imgSize(string $imgUrl): bool
    {
        $params = $this->plugin->getParams();
        $minWidth = (int) $params->get('og_img_min_width', 200); // Get min width, default 200
        $minHeight = (int) $params->get('og_img_min_height', 200); // Get min height, default 200

        // If minimums are set to 0 or less, skip the check
        if ($minWidth <= 0 && $minHeight <= 0) {
            return true;
        }

        // URL should already be absolute here, but check just in case
        if (strpos($imgUrl, 'http') !== 0) {
            $this->plugin->addOgDebug('Image Size Check Failed', 'Non-absolute URL passed: ' . $imgUrl, 'danger', 'exclamation-circle');
            return false; // Cannot check size of non-absolute URL reliably
        }


        // Attempt to get image size. Suppress errors as it might fail.
        $imageInfo = @getimagesize($imgUrl);

        if ($imageInfo === false || !isset($imageInfo[0]) || !isset($imageInfo[1])) {
            // Check if error reporting suggests a specific issue (e.g., allow_url_fopen disabled)
            $lastError = error_get_last();
             $errorMsg = $lastError ? ' - Error: ' . $lastError['message'] : '';
            $this->plugin->addOgDebug('Image Size Check Failed', 'Could not get dimensions for: ' . $imgUrl . $errorMsg, 'warning', 'exclamation-circle');
            // Decide policy: should we allow images if size check fails? Often safer to reject.
            return false; // Cannot determine size, assume invalid/too small
        }

        $width = (int) $imageInfo[0];
        $height = (int) $imageInfo[1];

        // Check against minimum dimensions
        $widthOk = ($minWidth <= 0) || ($width >= $minWidth);
        $heightOk = ($minHeight <= 0) || ($height >= $minHeight);

        if ($widthOk && $heightOk) {
            return true; // Meets requirements
        } else {
            $this->plugin->addOgDebug('Image Skipped (Size)', "{$width}x{$height} < {$minWidth}x{$minHeight} : " . $imgUrl, 'info', 'arrows-alt');
            return false; // Too small
        }
    }

    /**
     * Prepares an array of image URLs: cleans URLs, checks size, ensures uniqueness, limits quantity.
     * Ensures returned URLs are absolute.
     *
     * @param array<string> $images       Array of raw image URLs.
     * @param int|null      $maxImages    Maximum number of images to return. Null uses plugin setting.
     * @param bool          $skipSizeCheck If true, bypasses the minimum size check (e.g., for default image).
     *
     * @return array<string> Array of prepared, absolute image URLs.
     */
    private function prepareImages(array $images, ?int $maxImages = null, bool $skipSizeCheck = false): array
    {
        if ($maxImages === null) {
             $params = $this->plugin->getParams();
             $maxImages = ($params->get('og_multi_img', '0') == '0')
                 ? 1
                 : (int) $params->get('og_max_multi_img', 3);
        }

        $images_ok = [];

        foreach ($images as $image) {
             if (!is_string($image) || empty(trim($image))) {
                 $this->plugin->addOgDebug('Prepare Image Skip', 'Invalid input type or empty string.', 'warning', 'chain-broken');
                 continue;
             }

            // Clean the URL (makes absolute based on site root, basic validation)
            // HTMLHelper::_ cleans the URL, but let's be explicit about checking absoluteness
            $cleanedImage = HTMLHelper::cleanImageURL($image);

            if ($cleanedImage && !empty($cleanedImage->url)) {
                $absoluteUrl = $cleanedImage->url;

                // Double-check if it's truly absolute (starts with http or https)
                // cleanImageURL should handle this, but extra check for safety.
                if (strpos($absoluteUrl, 'http') !== 0) {
                     // If still not absolute, attempt to prepend site base URI
                     // This might happen if cleanImageURL fails for some reason or if the path is very unusual
                      $absoluteUrl = Uri::base() . ltrim($absoluteUrl, '/');
                      $this->plugin->addOgDebug('Prepare Image Warning', 'URL forced absolute: ' . $absoluteUrl, 'warning', 'link');
                 }


                // Check size unless skipped OR size check passes
                if ($skipSizeCheck || $this->imgSize($absoluteUrl)) { // Pass the confirmed absolute URL to imgSize
                    $images_ok[] = $absoluteUrl; // Store the absolute URL
                }
                // else: Debug message for size skip happens inside imgSize()
            } else {
                $this->plugin->addOgDebug('Prepare Image Skip', 'Could not clean URL: ' . $image, 'warning', 'chain-broken');
            }
        }

        // Ensure uniqueness and re-index array
        $images_unique = array_unique($images_ok);
        $images_final = array_values($images_unique);

        // Limit to the maximum number required
        if ($maxImages > 0 && count($images_final) > $maxImages) {
            $images_final = array_slice($images_final, 0, $maxImages);
        }

        return $images_final;
    }

    /**
     * Checks if the default image should always be used and prepares it.
     *
     * @return array<string>|false Array containing the default image URL if 'Always' setting is active, otherwise false.
     */
    private function prepareDefAlways()
    {
        $params = $this->plugin->getParams();

        // Check if option '1' (Always use default) is selected
        if ($params->get('opengraph_default_image_opt', '0') == '1') {
            $defaultImg = $params->get('opengraph_default_image', '');
            if (!empty($defaultImg)) {
                // Prepare the default image (clean URL, maybe skip size check)
                 $preparedDefault = $this->prepareImages([$defaultImg], 1, true); // Skip size check for 'Always' default
                if (!empty($preparedDefault)) {
                    $this->plugin->addOgDebug('Using Default Image (Always)', $preparedDefault[0], 'success', 'image');
                    return $preparedDefault; // Return array with the single default image
                } else {
                     $this->plugin->addOgDebug('Default Image Failed (Always)', 'URL: ' . $defaultImg, 'danger', 'exclamation-triangle');
                     return false; // Default image specified but failed preparation
                }
            } else {
                 $this->plugin->addOgDebug('Default Image Missing (Always)', 'Setting enabled but no URL provided.', 'warning', 'exclamation-triangle');
                 return false; // 'Always' selected but no image provided
            }
        }

        return false; // 'Always' setting not active
    }
}
