<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  Content.social2s
 *
 * @copyright   Copyright (C) 2005 - 2023 jtotal.org All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Jtotal\Plugin\Content\Social2s\Helper;

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Uri\Uri;
use Jtotal\Plugin\Content\Social2s\Extension\Social2s; // Keep for type hinting

defined('_JEXEC') or die;

/**
 * Helper class for generating individual social share button HTML.
 */
class S2sSocial
{
    /**
     * The main plugin instance.
     * @var Social2s
     */
    protected $plugin;

    /**
     * Prepared data for the current article/item.
     * @var object
     */
    protected $articleData;

    /**
     * Pre-rendered HTML templates for buttons (icon/img + text).
     * @var array<string, string>
     */
    protected $buttonTemplates;

    /**
     * Undocumented variable
     *
     * @var object
     */
    protected $styles;

    /**
     * Constructor.
     *
     * @param Social2s $plugin          The main Social2s plugin instance.
     * @param object   $articleData     Prepared data object from S2sTmpl.
     * @param array    $buttonTemplates Pre-rendered button content templates.
     */
    public function __construct(Social2s $plugin, object $articleData, array $buttonTemplates, object $styles)
    {
        $this->plugin = $plugin;
        $this->articleData = $articleData;
        $this->buttonTemplates = $buttonTemplates;
        $this->styles = $styles;
    }

    /**
     * Generates Twitter (X) share button HTML.
     *
     * @param string $cssClass Additional CSS class(es) (e.g., 's2s-in-more').
     * @return string HTML output.
     */
    public function getTwitter(string $cssClass = ''): string
    {
        $params = $this->plugin->getParams();

        $s2stmpl = new S2sTmpl($this->plugin);
        $styles = $this->styles; // Assuming getter exists or get directly

        // --- Twitter Specific Params ---
        $twitter_user = $params->get('twitter_user', '');
        $twitter_via = $params->get('twitter_via', 0);
        // Counter deprecated/removed $twitter_counter = $params->get('twitter_count', 0);
        // Follow button within globo removed $twitter_follow = $params->get('twitter_follow', 0);

        // --- Build Share URL ---
        $viaParam = ($twitter_via && !empty($twitter_user)) ? '&via=' . ltrim($twitter_user, '@') : '';
        $shareUrl = 'https://twitter.com/intent/tweet?text=' . $this->articleData->safetitle
            . '&url=' . $this->articleData->encoded_full_link
            . $viaParam;

        // --- Prepare Output ---
        $tabindex = ($cssClass === 's2s-in-more') ? '-1' : '0'; // Make non-interactive if hidden initially
        $buttonContent = $this->buttonTemplates['twitter'] ?? 'X'; // Fallback text

        $html = '<div class="s2s_twitter s2s_btn ' . ($styles->size ?? 's2s-btn-default') . ' ' . trim($cssClass) . '">';
        $html .= '<a
                    aria-label="' . Text::_('SOCIAL2S_SHARE_ON_TWITTER') . '"
                    role="button"
                    tabindex="' . $tabindex . '"
                    href="' . htmlspecialchars($shareUrl) . '"
                    onclick="javascript:s2sClick(this,event); return false;"
                    class="s2s_icon">'
            . $buttonContent
            . '</a>';
        $html .= '</div>';

        return $html;
    }


    /**
     * Generates Facebook share button HTML.
     *
     * @param string $cssClass Additional CSS class(es).
     * @return string HTML output.
     */
    public function getFacebook(string $cssClass = ''): string
    {
        $params = $this->plugin->getParams();
        $styles = $this->styles;

        // --- Facebook Specific Params ---
        // Counters deprecated $facebook_share_count = $params->get('facebook_share_count', 0);
        // $facebook_like_count = $params->get('facebook_like_count', 0);

        // --- Build Share URL ---
        // Basic FB Sharer URL - Note: FB SDK offers more features but requires JS init
        $shareUrl = 'https://www.facebook.com/sharer/sharer.php?u=' . $this->articleData->encoded_full_link;

        // --- Prepare Output ---
        $tabindex = ($cssClass === 's2s-in-more') ? '-1' : '0';
        $buttonContent = $this->buttonTemplates['facebook'] ?? 'Facebook';

        $html = '<div class="s2s_facebook s2s_btn ' . ($styles->size ?? 's2s-btn-default') . ' ' . trim($cssClass) . '">';
        $html .= '<a
                    aria-label="' . Text::_('SOCIAL2S_SHARE_ON_FACEBOOK') . '"
                    role="button"
                    tabindex="' . $tabindex . '"
                    href="' . htmlspecialchars($shareUrl) . '"
                    onclick="javascript:s2sClick(this,event); return false;"
                    class="s2s_icon">'
            . $buttonContent
            // Add counter spans here if needed and if counts are fetched via JS
            . '</a>';
        $html .= '</div>';

        return $html;
    }

    /**
     * Generates Pinterest "Pin It" button HTML.
     *
     * @param string $cssClass Additional CSS class(es).
     * @return string HTML output.
     */
    public function getPinterest(string $cssClass = ''): string
    {
        $params = $this->plugin->getParams();
        $styles = $this->styles;

        // --- Pinterest Specific Params ---
        // Counts deprecated/removed

        // --- Build Share URL ---
        // Pinterest requires media URL. Use the first image found.
        $mediaUrl = $this->articleData->images[0] ?? ''; // Get first image URL
        if (empty($mediaUrl)) {
            $this->plugin->addDebug('Pinterest Button', 'Skipped: No image found for pinning.', 'warning');
            return ''; // Cannot pin without an image
        }

        $shareUrl = 'https://www.pinterest.com/pin/create/button/?url=' . $this->articleData->encoded_full_link
            . '&media=' . rawurlencode($mediaUrl) // Important: Add media URL
            . '&description=' . $this->articleData->safetitle;

        // --- Prepare Output ---
        $tabindex = ($cssClass === 's2s-in-more') ? '-1' : '0';
        $buttonContent = $this->buttonTemplates['pinterest'] ?? 'Pinterest';

        $html = '<div class="s2s_pinterest s2s_btn ' . ($styles->size ?? 's2s-btn-default') . ' ' . trim($cssClass) . '">';
        $html .= '<a
                    aria-label="' . Text::_('SOCIAL2S_PIN_IT') . '"
                    role="button"
                    tabindex="' . $tabindex . '"
                    href="' . htmlspecialchars($shareUrl) . '"
                    onclick="javascript:s2sClick(this,event); return false;"
                    data-pin-do="buttonBookmark"
                    class="s2s_icon">' // data-pin-do might be needed for Pinterest JS
            . $buttonContent
            . '</a>';
        $html .= '</div>';

        return $html;
    }

    /**
     * Generates LinkedIn share button HTML.
     *
     * @param string $cssClass Additional CSS class(es).
     * @return string HTML output.
     */
    public function getLinkedin(string $cssClass = ''): string
    {
        $params = $this->plugin->getParams();
        $styles = $this->styles;

        // --- Build Share URL ---
        // Basic LinkedIn Sharer URL
        $shareUrl = 'https://www.linkedin.com/shareArticle?mini=true&url=' . $this->articleData->encoded_full_link
            . '&title=' . $this->articleData->safetitle
            . '&summary=' . rawurlencode($this->articleData->description ?? '') // Optional summary
            . '&source=' . rawurlencode($this->plugin->getSiteName()); // Optional source

        // --- Prepare Output ---
        $tabindex = ($cssClass === 's2s-in-more') ? '-1' : '0';
        $buttonContent = $this->buttonTemplates['linkedin'] ?? 'LinkedIn';

        $html = '<div class="s2s_linkedin s2s_btn ' . ($styles->size ?? 's2s-btn-default') . ' ' . trim($cssClass) . '">';
        $html .= '<a
                    aria-label="' . Text::_('SOCIAL2S_SHARE_ON_LINKEDIN') . '"
                    role="button"
                    tabindex="' . $tabindex . '"
                    href="' . htmlspecialchars($shareUrl) . '"
                    onclick="javascript:s2sClick(this,event); return false;"
                    class="s2s_icon">'
            . $buttonContent
            . '</a>';
        $html .= '</div>';

        return $html;
    }

    /**
     * Generates WhatsApp share button HTML (sharing URL only).
     *
     * @param string $cssClass Additional CSS class(es).
     * @return string HTML output.
     */
    public function getWapp(string $cssClass = ''): string
    {
        $params = $this->plugin->getParams();
        $styles = $this->styles; // Use styles passed to constructor
        $buttonSize = $styles->size ?? 's2s-btn-default';

        // --- WhatsApp Specific Params ---
        $onlyMobile = $params->get('wapp_only_mobile', 0);
        $mobileClass = $onlyMobile ? 's2s_mobile_only' : '';

        // --- Build Share URL ---
        // URL format for WhatsApp link - text ONLY includes the URL
        $shareText = rawurlencode($this->articleData->full_link); // <-- Use URL only
        $shareUrl = 'https://wa.me/?text=' . $shareText; // Web API link
        $shareUrlMobile = 'whatsapp://send?text=' . $shareText; // Mobile app link

        // --- Prepare Output ---
        $tabindex = ($cssClass === 's2s-in-more') ? '-1' : '0';
        $buttonContent = $this->buttonTemplates['wapp'] ?? 'WhatsApp';

        $html = '<div class="s2s_wapp s2s_btn ' . $buttonSize . ' ' . $mobileClass . ' ' . trim($cssClass) . '">';
        // Use web link (wa.me) as primary href for broader compatibility
        $html .= '<a
                    aria-label="' . Text::_('SOCIAL2S_SHARE_ON_WAPP') . '"
                    role="button"
                    tabindex="' . $tabindex . '"
                    href="' . htmlspecialchars($shareUrl) . '" ' . // Use wa.me link primarily
                    'data-mobile-url="' . htmlspecialchars($shareUrlMobile) . '"' . // Store mobile protocol link if JS wants to use it
                    'data-action="share/whatsapp/share"' . // Keep if JS uses it
                    // Remove data-text as it's not part of the URL protocols
                    // 'data-text="' . htmlspecialchars($this->articleData->title) . '"' . // REMOVED/COMMENTED
                    'onclick="javascript:s2sWappClick(this,event); return false;"' . // JS might handle switching to mobile URL
                    'target="_blank"' . // Good practice for external links/apps
                    'rel="noopener noreferrer"' . // Security for target="_blank"
                    ' class="s2s_icon">'
                 . $buttonContent
                 . '</a>';
        $html .= '</div>';

        return $html;
    }

    /**
     * Generates Telegram share button HTML.
     *
     * @param string $cssClass Additional CSS class(es).
     * @return string HTML output.
     */
    public function getTgram(string $cssClass = ''): string
    {
        $params = $this->plugin->getParams();
        //$styles = $this->plugin->getCalculatedStyles();

        // --- Telegram Specific Params ---
        $onlyMobile = $params->get('tgram_only_mobile', 0);
        $mobileClass = $onlyMobile ? 's2s_mobile_only' : '';



        // --- Build Share URL ---
        $shareUrl = 'https://telegram.me/share/url?url=' . $this->articleData->encoded_full_link
            . '&text=' . $this->articleData->safetitle;

        // --- Prepare Output ---
        $tabindex = ($cssClass === 's2s-in-more') ? '-1' : '0';
        $buttonContent = $this->buttonTemplates['tgram'] ?? 'Telegram';

        $html = '<div class="s2s_tgram s2s_btn ' . ($this->styles->size ?? 's2s-btn-default') . ' ' . $mobileClass . ' ' . trim($cssClass) . '">';
        $html .= '<a
                    aria-label="' . Text::_('SOCIAL2S_SHARE_ON_TGRAM') . '"
                    role="button"
                    tabindex="' . $tabindex . '"
                    href="' . htmlspecialchars($shareUrl) . '"
                    target="_blank"  
                    rel="noopener noreferrer"
                    class="s2s_icon">'
            . $buttonContent
            . '</a>';
        $html .= '</div>';

        return $html;
    }

    /**
     * Generates Flipboard share button HTML.
     *
     * @param string $cssClass Additional CSS class(es).
     * @return string HTML output.
     */
    public function getFlipb(string $cssClass = ''): string
    {
        $params = $this->plugin->getParams();
        //$styles = $this->plugin->getCalculatedStyles();

        // --- Flipboard Specific Params ---
        $onlyMobile = $params->get('flipb_only_mobile', 0);
        $mobileClass = $onlyMobile ? 's2s_mobile_only' : '';

        // --- Build Share URL ---
        $shareUrl = 'https://share.flipboard.com/bookmarklet/popout?v=2&title=' . $this->articleData->safetitle
            . '&url=' . $this->articleData->encoded_full_link;
        // Original included &t=timestamp&utm... - These are likely analytics/tracking, potentially omit unless required by Flipboard API.
        // . '&t=' . time() . '&utm_campaign=tools&utm_medium=article-share';

        // --- Prepare Output ---
        $tabindex = ($cssClass === 's2s-in-more') ? '-1' : '0';
        $buttonContent = $this->buttonTemplates['flipb'] ?? 'Flipboard';

        $html = '<div class="s2s_flipb s2s_btn ' . ($this->styles->size ?? 's2s-btn-default') . ' ' . $mobileClass . ' ' . trim($cssClass) . '">';
        $html .= '<a
                    aria-label="' . Text::_('SOCIAL2S_SHARE_ON_FLIPB') . '"
                    role="button"
                    tabindex="' . $tabindex . '"
                    href="' . htmlspecialchars($shareUrl) . '"
                    onclick="javascript:s2sClick(this,event); return false;"
                    class="s2s_icon">'
            . $buttonContent
            . '</a>';
        $html .= '</div>';

        return $html;
    }

    /**
     * Generates Snapchat share button HTML.
     * Note: Snapchat sharing usually involves scanning a Snapcode or deep linking,
     *       sharing a URL directly via a button is less common. This uses the scan link.
     *
     * @param string $cssClass Additional CSS class(es).
     * @return string HTML output.
     */
    public function getSnap(string $cssClass = ''): string
    {
        $params = $this->plugin->getParams();
        //$styles = $this->plugin->getCalculatedStyles();

        // --- Snapchat Specific Params ---
        $onlyMobile = $params->get('snap_only_mobile', 0);
        $mobileClass = $onlyMobile ? 's2s_mobile_only' : '';

        // --- Build Share URL ---
        // URL to open the Snapchat app with the article URL as an attachment (works primarily on mobile)
        $shareUrl = 'https://snapchat.com/scan?attachmentUrl=' . $this->articleData->encoded_full_link
            . '&utm_source=' . rawurlencode($this->plugin->getSiteName());

        // --- Prepare Output ---
        $tabindex = ($cssClass === 's2s-in-more') ? '-1' : '0';
        $buttonContent = $this->buttonTemplates['snap'] ?? 'Snapchat';

        $html = '<div class="s2s_snap s2s_btn ' . ($this->styles->size ?? 's2s-btn-default') . ' ' . $mobileClass . ' ' . trim($cssClass) . '">';
        $html .= '<a
                    aria-label="' . Text::_('SOCIAL2S_SHARE_ON_SNAP') . '"
                    role="button"
                    tabindex="' . $tabindex . '"
                    href="' . htmlspecialchars($shareUrl) . '"
                    target="_blank" 
                    rel="noopener noreferrer"
                    class="s2s_icon">'
            . $buttonContent
            . '</a>';
        $html .= '</div>';

        return $html;
    }


    /**
     * Generates Reddit share button HTML.
     *
     * @param string $cssClass Additional CSS class(es).
     * @return string HTML output.
     */
    public function getReddit(string $cssClass = ''): string
    {
        $params = $this->plugin->getParams();
        //$styles = $this->plugin->getCalculatedStyles();

        // --- Build Share URL ---
        $shareUrl = 'https://www.reddit.com/submit?url=' . $this->articleData->encoded_full_link
            . '&title=' . $this->articleData->safetitle;

        // --- Prepare Output ---
        $tabindex = ($cssClass === 's2s-in-more') ? '-1' : '0';
        $buttonContent = $this->buttonTemplates['reddit'] ?? 'Reddit';

        $html = '<div class="s2s_reddit s2s_btn ' . ($this->styles->size ?? 's2s-btn-default') . ' ' . trim($cssClass) . '">';
        $html .= '<a
                    aria-label="' . Text::_('SOCIAL2S_SHARE_ON_REDDIT') . '"
                    role="button"
                    tabindex="' . $tabindex . '"
                    href="' . htmlspecialchars($shareUrl) . '"
                    onclick="javascript:s2sClick(this,event); return false;"
                    class="s2s_icon">'
            . $buttonContent
            . '</a>';
        $html .= '</div>';

        return $html;
    }

    /**
     * Generates Tumblr share button HTML.
     *
     * @param string $cssClass Additional CSS class(es).
     * @return string HTML output.
     */
    public function getTumblr(string $cssClass = ''): string
    {
        $params = $this->plugin->getParams();
        //$styles = $this->plugin->getCalculatedStyles();

        // --- Tumblr Specific Params ---
        // Counts deprecated/removed

        // --- Build Share URL ---
        $shareUrl = 'https://www.tumblr.com/share/link?url=' . $this->articleData->encoded_full_link
            . '&name=' . $this->articleData->safetitle // Use 'name' for title
            . '&description=' . rawurlencode($this->articleData->description ?? ''); // Optional description

        // --- Prepare Output ---
        $tabindex = ($cssClass === 's2s-in-more') ? '-1' : '0';
        $buttonContent = $this->buttonTemplates['tumblr'] ?? 'Tumblr';
        // Add counter spans here if JS fetches counts $tumblr_s_count_html = '<span class="s2s_badge s2s_tumblr_badge_count">0</span>';

        $html = '<div class="s2s_tumblr s2s_btn ' . ($this->styles->size ?? 's2s-btn-default') . ' ' . trim($cssClass) . '">';
        $html .= '<a
                    aria-label="' . Text::_('SOCIAL2S_SHARE_ON_TUMBLR') . '"
                    role="button"
                    tabindex="' . $tabindex . '"
                    href="' . htmlspecialchars($shareUrl) . '"
                    onclick="javascript:s2sClick(this,event); return false;"
                    class="s2s_icon">'
            . $buttonContent
            // . $tumblr_s_count_html // Add counter here if used
            . '</a>';
        $html .= '</div>';

        return $html;
    }

    /**
     * Generates VK share button HTML.
     *
     * @param string $cssClass Additional CSS class(es).
     * @return string HTML output.
     */
    public function getVk(string $cssClass = ''): string
    {
        $params = $this->plugin->getParams();
        //$styles = $this->plugin->getCalculatedStyles();

        // --- Build Share URL ---
        // VK share URL structure
        $shareUrl = 'https://vk.com/share.php?url=' . $this->articleData->encoded_full_link
            . '&title=' . $this->articleData->safetitle
            // Add image if available and desired
            . (!empty($this->articleData->images[0]) ? '&image=' . rawurlencode($this->articleData->images[0]) : '')
            . '&description=' . rawurlencode($this->articleData->description ?? '')
            . '&noparse=true'; // Often helps ensure parameters are used


        // --- Prepare Output ---
        $tabindex = ($cssClass === 's2s-in-more') ? '-1' : '0';
        $buttonContent = $this->buttonTemplates['vk'] ?? 'VK';

        $html = '<div class="s2s_vk s2s_btn ' . ($this->styles->size ?? 's2s-btn-default') . ' ' . trim($cssClass) . '">';
        $html .= '<a
                    aria-label="' . Text::_('SOCIAL2S_SHARE_ON_VK') . '"
                    role="button"
                    tabindex="' . $tabindex . '"
                    href="' . htmlspecialchars($shareUrl) . '"
                    onclick="javascript:s2sClick(this,event); return false;"
                    class="s2s_icon">'
            . $buttonContent
            . '</a>';
        $html .= '</div>';

        return $html;
    }

    /**
     * Generates Email ("mailto:") share button HTML.
     *
     * @param string $cssClass Additional CSS class(es).
     * @return string HTML output.
     */
    public function getEmail(string $cssClass = ''): string
    {
        $params = $this->plugin->getParams();
        //$styles = $this->plugin->getCalculatedStyles();

        // --- Build Mailto URL ---
        $subject = rawurlencode($this->articleData->title);
        // Body text: include title and URL
        $body = rawurlencode($this->articleData->title . "\n\n" . $this->articleData->full_link);
        $shareUrl = 'mailto:?subject=' . $subject . '&body=' . $body;

        // --- Prepare Output ---
        $tabindex = ($cssClass === 's2s-in-more') ? '-1' : '0';
        $buttonContent = $this->buttonTemplates['email'] ?? 'Email';

        $html = '<div class="s2s_email s2s_btn ' . ($this->styles->size ?? 's2s-btn-default') . ' ' . trim($cssClass) . '">';
        $html .= '<a
                    aria-label="' . Text::_('SOCIAL2S_SHARE_VIA_EMAIL') . '"
                    role="button"
                    tabindex="' . $tabindex . '"
                    href="' . htmlspecialchars($shareUrl) . '"
                    class="s2s_icon">' // No JS click handler needed for mailto usually
            . $buttonContent
            . '</a>';
        $html .= '</div>';

        return $html;
    }

    /**
     * Generates the "More" button HTML trigger ONLY.
     * Visibility of other buttons relies on CSS targeting '.s2s-after-more'.
     *
     * @param array $buttonsInMore (Unused in this version, kept for signature consistency if needed elsewhere, but logic ignores it)
     * @return string HTML output for the 'more' button trigger.
     */
    public function getMore(array $buttonsInMore = []): string // Parameter $buttonsInMore is now ignored
    {
        $params = $this->plugin->getParams();
        $styles = $this->styles; // Use styles passed to constructor

        // --- "More" Button Specific Params ---
        $s2s_icon_image_type = $params->get('s2s_icon_image', 'fa');
        $mas_style_setting = $params->get('mas_style', 'plus-square');
        $s2s_icon_image_custom = $params->get('mas_style_custom', 'fas fa-plus-square');
        $onlyMobile = ($params->get('mas_on', 0) == '2');
        $mobileClass = $onlyMobile ? 's2s_mobile_only' : '';
        $image_pixels = $styles->image_size ?? '24px';

        // --- Determine Icon HTML for the trigger ---
        $masButtonIcon = '';
        $plusIcon = '';
        $minusIcon = '';

        // ... (Keep the exact same logic as before to determine $plusIcon and $minusIcon based on settings) ...
        if ($s2s_icon_image_type === 'fa') { /* ... FA logic ... */
            switch ($mas_style_setting) {
                case 'plus-circle':
                    $plusIcon = '<i class="fas fa-plus-circle"></i>';
                    $minusIcon = '<i class="fas fa-minus-circle"></i>';
                    break;
                case 'plus_custom':
                    $plusIcon = '<i class="' . htmlspecialchars($s2s_icon_image_custom) . '"></i>';
                    $minusIcon = '<i class="fas fa-minus-square"></i>';
                    break;
                case 'plus-square':
                default:
                    $plusIcon = '<i class="fas fa-plus-square"></i>';
                    $minusIcon = '<i class="fas fa-minus-square"></i>';
                    break;
            }
        } elseif ($s2s_icon_image_type === 'image') { /* ... Image logic with quoted alts ... */
            $imgBase = $this->plugin->getMediaPath() . '/assets/svg/';
            switch ($mas_style_setting) {
                case 'plus-circle':
                    $plusIcon = '<img loading="lazy" class="s2s-plus-icon" src="' . $imgBase.'plus-circle-solid.svg"';
                    $plusIcon .=  ' alt="More" width="' . $image_pixels . '" height="' . $image_pixels . '"/>';
                    $minusIcon = '<img loading="lazy" class="s2s-minus-icon" src="' . $imgBase . 'minus-circle-solid.svg" ';
                    $minusIcon .= ' alt="Less" width="' . $image_pixels . '" height="' . $image_pixels . '"/>';
                    break;
                case 'plus_custom':
                    if (strpos($s2s_icon_image_custom, '<') !== false) {
                        $plusIcon = $s2s_icon_image_custom;
                    } else {
                        $plusIcon = '<i class="' . htmlspecialchars($s2s_icon_image_custom) . '"></i>';
                    }
                    $minusIcon = '<img loading="lazy" class="s2s-minus-icon" src="' . $imgBase . 'minus-square-solid.svg" alt="Less" width="' . $image_pixels . '" height="' . $image_pixels . '"/>';
                    break;
                case 'plus-square':
                default:
                    $plusIcon = '<img loading="lazy" class="s2s-plus-icon" src="' . $imgBase . 'plus-square-solid.svg" alt="More" width="' . $image_pixels . '" height="' . $image_pixels . '"/>';
                    $minusIcon = '<img loading="lazy" class="s2s-minus-icon" src="' . $imgBase . 'minus-square-solid.svg" alt="Less" width="' . $image_pixels . '" height="' . $image_pixels . '"/>';
                    break;
            }
        } else {
            $plusIcon = '+';
            $minusIcon = '-';
        }
            $masButtonIcon = '<span class="s2s-mas-icon-open">' . $plusIcon . '</span><span class="s2s-mas-icon-close">' . $minusIcon . '</span>';


        // --- Prepare Output ---
        // Just the trigger button wrapper and the trigger link
        $html = '<div class="s2s_mas s2s_btn ' . ($styles->size ?? 's2s-btn-default') . ' ' . $mobileClass . ' s2s_mas_style_' . $s2s_icon_image_type . '">';
        $html .= '<a href="#"
                    aria-label="' . Text::_('SOCIAL2S_MORE_OPTIONS') . '"
                    aria-expanded="false"
                    role="button"
                    tabindex="0"
                    onclick="javascript:s2sToggleMore(this, event); return false;" ' // JS handler toggles visibility of '.s2s-after-more'
                    . ' class="s2s_mas_trigger">'
                 . $masButtonIcon
                 . '</a>';
        // NO hidden container div here anymore
        $html .= '</div>'; // End s2s_mas

        return $html;
    }


    /**
     * Gets the localized "Tweet" text for the Twitter button.
     *
     * @param string $lang Short language code (e.g., 'en', 'es').
     * @return string Localized text wrapped in span, or default English text.
     */
    private function getTwitterLanguage(string $lang): string
    {
        // Kept original array structure for direct lookup
        $tweet_lang = [
            'en' => 'Tweet',
            'ar' => 'غرِّد',
            'bn' => 'টুইট',
            'cs' => 'Tweet',
            'da' => 'Tweet',
            'ca' => 'Tuit',
            'de' => 'Twittern',
            'el' => 'Tweet',
            'es' => 'Twittear',
            'fa' => 'توییت',
            'fi' => 'Twiittaa',
            'fil' => 'I-tweet',
            'fr' => 'Tweeter',
            'he' => 'ציוץ',
            'hi' => 'ट्वीट',
            'hu' => 'Tweet',
            'id' => 'Tweet',
            'it' => 'Tweet',
            'ja' => 'ツイート',
            'ko' => '트윗',
            'msa' => 'Tweet',
            'nl' => 'Tweeten',
            'no' => 'Tweet',
            'pl' => 'Tweetnij',
            'pt' => 'Tweetar',
            'ro' => 'Tweet',
            'ru' => 'Твитнуть',
            'sv' => 'Tweeta',
            'th' => 'ทวีต',
            'tr' => 'Tweetle',
            'uk' => 'Твіт',
            'ur' => 'ٹویٹ کریں',
            'vi' => 'Tweet',
            'zh-cn' => '发推',
            'zh-tw' => '推文',
        ];

        $text = $tweet_lang[$lang] ?? $tweet_lang['en'];

        // Return text only, wrapping span is handled by getButtonTemplates/getTexts
        return $text;
        // If used directly: return '<span>' . htmlspecialchars($text) . '</span>';
    }

    /**
     * Renders the ordered list of enabled social buttons based on plugin settings.
     * The 'more' button trigger, if enabled, is always placed visually last.
     * Buttons appearing after the 'more' trigger's position in the order setting
     * get the 's2s-after-more' class for CSS/JS visibility control.
     *
     * @return string The generated HTML for all enabled buttons in the correct order, with 'more' last.
     */
    public function renderOrderedButtons(): string
    {
        // 1. Get parameters and order setting
        $params = $this->plugin->getParams();
        $orderSetting = $params->get('s2s_order', '0,1,2,3,4,5,6,7,8,9,10,11,12,13');
        $this->plugin->addDebug('Button Order Raw', "'" . $orderSetting . "'", 'info', 'list-ol');
        $s2s_order = array_filter(array_map('trim', explode(',', $orderSetting)), 'strlen');
        $this->plugin->addDebug('Button Order Array', implode(', ', $s2s_order), 'info', 'list-ol');

        // 2. Define mappings
        $orderMap = [
            '0' => 'twitter', '1' => 'facebook', '2' => 'pinterest', '3' => 'linkedin',
            '4' => null, '5' => 'wapp', '6' => 'tgram', '7' => 'flipb', '8' => 'snap',
            '9' => 'tumblr', '10' => 'vk', '11' => 'email', '12' => 'reddit', '13' => 'more'
        ];
        $networkParams = [
            'twitter' => 'twitter_on', 'facebook' => 'facebook_on', 'pinterest' => 'pinterest_on',
            'linkedin' => 'linkedin_on', 'wapp' => 'wapp_on', 'tgram' => 'tgram_on',
            'flipb' => 'flipb_on', 'snap' => 'snap_on', 'tumblr' => 'tumblr_on',
            'vk' => 'vk_on', 'email' => 'email_on', 'reddit' => 'reddit_on',
            'more' => 'mas_on' // Parameter for enabling the 'more' functionality
        ];

        // 3. Determine if the 'more' functionality is active
        $isMoreFeatureEnabled = false;
        $param_mas_on = $params->get('mas_on', 0);
        if ($param_mas_on == 1 || ($param_mas_on == 2 && ($this->plugin->isMobileDetected ?? false))) {
            $isMoreFeatureEnabled = true;
        }
        $this->plugin->addDebug('More Button Logic', 'FeatureEnabled='.(int)$isMoreFeatureEnabled, 'info');

        // 4. Initialize variables
        $html = '';
        $moreButtonHtml = ''; // Store the 'more' button HTML separately
        $moreTriggerPositionPassed = false; // Flag to track if we are past the 'more' button position in the defined order

        // 5. Loop through the defined order
        foreach ($s2s_order as $index => $orderKey) {
            $networkKey = $orderMap[$orderKey] ?? null;
            if ($networkKey === null) {
                continue;
            }

            $paramName = $networkParams[$networkKey] ?? null;

            // Determine if this specific button should be rendered
            $isButtonEnabled = false;
            if ($networkKey === 'more') {
                // Only consider the 'more' *trigger* if the feature is enabled
                $isButtonEnabled = $isMoreFeatureEnabled;
            } elseif ($paramName !== null) {
                $isButtonEnabled = (bool) $params->get($paramName, 1);
            } else {
                continue; // Skip unknown network
            }

            if ($isButtonEnabled) {
                $methodName = 'get' . ucfirst($networkKey); // e.g., getTwitter, getMore

                if (method_exists($this, $methodName)) {
                    // Determine if the 'after-more' class should be added
                    // Add it if the 'more' feature is enabled AND we have passed the trigger position IN THE USER-DEFINED ORDER
                    $cssClass = ($isMoreFeatureEnabled && $moreTriggerPositionPassed) ? 's2s-after-more' : '';

                    if ($networkKey === 'more') {
                        // Generate the 'more' trigger button HTML BUT store it instead of appending
                        $this->plugin->addDebug('Button Loop Action', 'Generating "More" (' . $orderKey . ') trigger HTML', 'info');
                        $moreButtonHtml = $this->$methodName([]); // Call getMore to get its HTML
                        // Mark that we have now passed the trigger position in the order list
                        $moreTriggerPositionPassed = true;
                    } else {
                        // Render a regular social button, passing the calculated CSS class, append to main HTML
                        $this->plugin->addDebug('Button Loop Action', 'Rendering ' . $networkKey . ' (' . $orderKey . ') with class: ' . $cssClass, 'info');
                        $html .= $this->$methodName($cssClass);
                    }
                } else {
                     $this->plugin->addDebug('Button Generation Error', 'Method not found in S2sSocial: ' . $methodName, 'warning');
                }
            } else {
                 $this->plugin->addDebug('Button Loop Skip', 'Button ' . $networkKey . ' (' . $orderKey . ') is disabled.', 'info');
            }
        } // end foreach

        // 6. Append the stored "More" button HTML at the very end, if it was generated
        if (!empty($moreButtonHtml)) {
             $html .= $moreButtonHtml;
             $this->plugin->addDebug('Button Order Final', '"More" button appended at the end.', 'info');
        }

        return $html;
    }
}
