<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  Content.social2s
 *
 * @copyright   Copyright (C) 2005 - 2022 jtotal.org All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Jtotal\Plugin\Content\Social2s\Helper;

use Joomla\CMS\Environment\Browser;
use Joomla\CMS\Factory;
use Joomla\CMS\Filter\OutputFilter;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;
use Jtotal\Plugin\Content\Social2s\Extension\Social2s; // Keep for type hinting

defined('_JEXEC') or die;

/**
 * Helper class for preparing data and generating HTML template output for Social2s.
 */
class S2sTmpl
{
    /**
     * The main plugin instance.
     * @var Social2s
     */
    protected $plugin;

    /**
     * Data extracted/prepared from the current article/item.
     * @var object|null
     */
    private $articleData = null;

    /**
     * Calculated style settings for buttons.
     * @var object|null
     */
    private $styles = null;

    /**
     * Calculated CSS classes for the main container.
     * @var object|null
     */
    private $supraClasses = null;

    /**
     * HTML content for each button type (icon + text).
     * @var array<string, string>
     */
    private $buttonTemplates = [];

    /**
     * Constructor.
     *
     * @param Social2s $plugin The main Social2s plugin instance.
     */
    public function __construct(Social2s $plugin)
    {
        $this->plugin = $plugin;
    }

    /**
     * Initializes data preparation, asset loading, and generates the plugin's display HTML.
     * This is the main entry point called by the plugin's event handlers.
     *
     * @param object $article The content object (article, product, event, etc.).
     *
     * @return string The generated HTML for the social buttons.
     */
    public function init(object $article): string
    {

        $this->articleData = $this->prepareArticleData($article);

        if ($this->articleData === null) {
             $this->plugin->addDebug(
                 'Template Init',
                 'Failed to prepare article data',
                 'danger',
                 'exclamation-triangle'
             );
             return ''; // Cannot proceed without data
        }

        $this->loadAssets();

        $this->setJavascriptVariables();

        $this->styles = $this->getCalculatedStyles();
        $this->supraClasses = $this->getSupraClasses();
        $this->buttonTemplates = $this->getButtonTemplates();

        $s2sSocialHelper = new S2sSocial($this->plugin, $this->articleData, $this->buttonTemplates, $this->styles);

        $buttonHtml = $s2sSocialHelper->renderOrderedButtons();

        $layoutData = [
            'plugin' => $this->plugin,
            'articleData' => $this->articleData,
            'styles' => $this->styles,
            'supraClasses' => $this->supraClasses,
            'buttonHtml' => $buttonHtml,
        ];

        $layoutPath = PluginHelper::getLayoutPath('content', 'social2s', 'default');

        if (file_exists($layoutPath)) {
            // Make the layout data available to the included file scope
            // (These variables will be directly usable inside default.php)
            $plugin       = $this->plugin;
            $articleData  = $this->articleData;
            $styles       = $this->styles;
            $supraClasses = $this->supraClasses;
            // $buttonHtml is already generated

            // Start output buffering
            ob_start();

            try {
                include $layoutPath; // Include the layout file
                $output = ob_get_clean(); // Get the buffered output
                $this->plugin->addDebug(
                    'Template Init',
                    'Rendered using layout: ' . basename($layoutPath),
                    'success',
                    'file-code-o'
                );
                return $output;
            } catch (\Exception $e) {
                ob_end_clean(); // Clean buffer on error during include
                $this->plugin->addDebug(
                    'Template Layout Error',
                    'Error including layout: ' . $e->getMessage(),
                    'danger',
                    'exclamation-triangle'
                );
                // Fallback rendering if include fails
                $wrapperClasses = htmlspecialchars(
                    $this->supraClasses->classes ?? 'social2s_wrapper',
                    ENT_QUOTES,
                    'UTF-8'
                );
                return '<div class="' . $wrapperClasses . '">' . $buttonHtml . '</div>';
            }
        } else {
            // Layout file not found - Fallback rendering
            $this->plugin->addDebug(
                'Template Layout Error',
                'Layout file not found at: ' . $layoutPath,
                'danger',
                'exclamation-triangle'
            );
            $wrapperClasses = htmlspecialchars($this->supraClasses->classes ?? 'social2s_wrapper', ENT_QUOTES, 'UTF-8');
            return '<div class="' . $wrapperClasses . '">' . $buttonHtml . '</div>';
        }
    }


    
    /**
     * Extracts relevant data (title, link, description, images) from the article object based on context.
     *
     * @param object $article The content object.
     * @return object|null An object containing prepared data, or null on failure.
     */
    private function prepareArticleData(object $article): ?object
    {

        $context = $this->plugin->getContext(); // e.g., com_content.category, com_content.article
        $app = Factory::getApplication();
        $uri = Uri::getInstance();
        $currentAbsoluteUrl = $uri->toString();

        // Default values - These apply to the specific item ($article) being processed
        $data = new \stdClass();
        $data->id = $article->id ?? null; // Article ID is crucial
        $data->catid = $article->catid ?? null; // Article's Category ID is crucial
        $data->title = $article->title ?? $app->getDocument()->getTitle();
        $data->slug = $article->slug ?? ($article->alias ?? null);
        $data->metadescription = $article->metadesc ?? $app->getDocument()->getMetaData('description'); // Prioritize article meta
        $data->description = '';
        $data->images = [];
        // Default links to article if possible, fallback to current page otherwise
        $data->link = $uri->toString(['path', 'query', 'fragment']);
        $data->full_link = $currentAbsoluteUrl;

        if (empty($data->title)) $data->title = $this->plugin->getSiteName();
        if (empty($data->slug) && !empty($data->title)) $data->slug = OutputFilter::stringURLSafe($data->title);

        $s2sImagesHelper = $this->plugin->getS2sImagesHelper();
        

        try {
            // Regardless of context (article, category, featured), if we have article data,
            // try to generate the ARTICLE's data first.
            $data->title = $article->title ?? $data->title;
            $data->description = $this->_descriptionFormatted($article->introtext ?? '');
            if(method_exists($s2sImagesHelper,'getImagesArticle')){
                $data->images = $s2sImagesHelper->getImagesArticle($article);
            }
             // Use specific article meta description if available
            if (!empty($article->metadesc)) $data->metadescription = $article->metadesc;

            // --- Generate ARTICLE link if possible (needs slug and catid from the article) ---
            if (!empty($data->slug) && !empty($data->catid)) {
                 $nonSefUrl = 'index.php?option=com_content&view=article&id=' . $data->slug . '&catid=' . $data->catid;
                 $data->link = Route::_($nonSefUrl); // Relative SEF article link
                 $data->full_link = Route::_($nonSefUrl, false, 2); // Absolute SEF article link
                 $this->plugin->addDebug('Prepare Data URL', 'Generated ARTICLE URL for item '.$data->id.' in context '.$context, 'info');
            } else {
                // Fallback if we couldn't get slug/catid from the $article object
                 $data->link = $uri->toString(['path', 'query', 'fragment']); // Use current page relative
                 $data->full_link = $currentAbsoluteUrl; // Use current page absolute
                 $this->plugin->addDebug('Prepare Data URL', 'Missing article slug/catid for item '.$data->id.' in context '.$context.', using CURRENT page URL.', 'warning');
            }

            switch ($context) {
                case 'com_virtuemart.productdetails':
                    // Override with VM specific data if needed (already handled article data above)
                    $data->title = $article->product_name ?? $data->title;
                    $data->description = $this->_descriptionFormatted($article->product_desc ?? $data->description); // Use VM desc, fallback to article desc
                    $data->images = $s2sImagesHelper->getImagesVirtuemart($article); // Get VM images
                    // Stick with current page URL for VM as routing is complex
                    $data->link = $uri->toString(['path', 'query', 'fragment']);
                    $data->full_link = $currentAbsoluteUrl;
                    $this->plugin->addDebug('Prepare Data VM Override', 'Using VM data, current URL.', 'info');
                    break;

                case 'com_jevents.event':
                    $data->title = $article->summary ?? $article->title ?? $data->title;
                    $data->description = $this->_descriptionFormatted($article->_description ?? $data->description);
                    $data->images = $s2sImagesHelper->getImagesJevents($article);
                    // Stick with current page URL for jEvents as routing is complex
                    $data->link = $uri->toString(['path', 'query', 'fragment']);
                    $data->full_link = $currentAbsoluteUrl;
                     $this->plugin->addDebug('Prepare Data jEvents Override', 'Using jEvents data, current URL.', 'info');
                    break;

                case 'com_k2.item':
                     $data->title = $article->title ?? $data->title;
                     $data->description = $this->_descriptionFormatted($article->introtext ?? $data->description);
                     $data->images = $s2sImagesHelper->getImagesK2($article);
                     // Stick with current page URL for K2 unless K2HelperRoute logic is proven reliable
                     $data->link = $uri->toString(['path', 'query', 'fragment']);
                     $data->full_link = $currentAbsoluteUrl;
                     $this->plugin->addDebug('Prepare Data K2 Override', 'Using K2 data, current URL.', 'info');
                    break;

                // No specific overrides needed for com_content.category/featured
                // as we already generated the ARTICLE URL above.
                case 'com_content.article':
                case 'com_content.category':
                case 'com_content.featured':
                     // Data already prepared using article properties and article URL routing.
                     break;

                default:
                    $this->plugin->addDebug('Prepare Data Context', 'Unknown context ['.$context.'], using article data defaults and current URL.', 'warning');
                    // Article data defaults and current URL were already set/attempted.
                    break;
            } // End Switch
        } catch (\Exception $e) {
            $this->plugin->addDebug('Prepare Data Error', 'Exception: ' . $e->getMessage(), 'danger', 'exclamation-triangle');
            // Ensure essential defaults before returning
            if (empty($data->title)) $data->title = $this->plugin->getSiteName();
            if (empty($data->full_link) || strpos($data->full_link, 'http') !== 0) $data->full_link = $currentAbsoluteUrl;
            $data->safetitle = rawurlencode($data->title);
            $data->encoded_full_link = rawurlencode($data->full_link);
            return $data; // Return best-effort data object
        }

        // Final cleanup/population
        if (empty($data->description) && !empty($data->metadescription)) {
            $data->description = $data->metadescription;
        }
        $data->safetitle = rawurlencode($data->title);
        $data->encoded_full_link = rawurlencode($data->full_link);

        // Final Logging (Log context along with final data)
        $this->plugin->addDebug('Final Title ['.$context.']', $data->title, 'success');
        $this->plugin->addDebug('Final Desc ['.$context.']', substr($data->description, 0, 100) . '...', 'success');
        $this->plugin->addDebug('Final Link ['.$context.']', $data->link, 'success');
        $this->plugin->addDebug('Final Full Link ['.$context.']', $data->full_link, 'success');
        $this->plugin->addDebug('Final Images ['.$context.']', count($data->images) . ' image(s)', 'success');

        return $data;
    }



    /**
     * Formats description text: removes plugin tags, strips HTML, limits length.
     *
     * @param string $text Input text.
     * @return string Formatted text.
     */
    private function _descriptionFormatted(string $text): string
    {
        // Remove {plugin tags} and {/plugin tags} more safely
        $text = preg_replace('/\{[^\}]+\}/', '', $text);
        // Strip HTML tags, allowing <br> initially if needed, then remove them
        $text = strip_tags($text); // Remove all tags directly
        // Remove line breaks
        $text = str_replace(["\r", "\n", "\r\n"], ' ', $text);
        // Decode HTML entities
        $text = html_entity_decode($text, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        // Limit length (e.g., to 200 chars for descriptions)
        $maxLength = 200;
        if (mb_strlen($text) > $maxLength) {
            $text = mb_substr($text, 0, $maxLength - 3) . '...';
        }
        // Trim whitespace
        return trim(preg_replace('/\s+/', ' ', $text)); // Collapse multiple spaces
    }

    /**
     * Sets mobile detection info.
     * Called internally by loadAssets or init if needed separately.
     */
    private function setMobileInfo(): void
    {
        // This logic seems fine, just make it an instance method accessing $this->plugin
        $browser = Browser::getInstance();
        $this->plugin->isMobileDetected = $browser->isMobile(); // Store on plugin if needed elsewhere

        // Add variable to JS if required by frontend script
        // Done in setJavascriptVariables now
    }

    /**
     * Loads required CSS and JavaScript assets using WebAssetManager.
     * Ensures assets are loaded only once per request.
     */
    private function loadAssets(): void
    {
        // Check if assets are already loaded for this request
        if ($this->plugin->assetsLoaded ?? false) {
             $this->plugin->addDebug('Assets', 'Already loaded.', 'info', 'check-square-o');
            return;
        }

        $params = $this->plugin->getParams();
        $wa = Factory::getDocument()->getWebAssetManager();
        $media_path = $this->plugin->getMediaPath(); // Base URL for media

        // Use unique version string for cache busting in debug mode or based on plugin version
        $version = ($params->get('s2s_debug', 0))
            ? md5((string) time()) // Force reload in debug
            : ($this->_getVersion() ?: '5.0.0'); // Use manifest version or fallback


        // --- Register Assets (using registry recommended if complex) ---
        // Assumes assets are registered via plg_content_social2s.xml or registry file
        $wa->getRegistry()->addExtensionRegistryFile('plg_content_social2s');


        // --- Use JavaScript ---
        $wa->useScript('plg_content_social2s.frontend')
            ->useScript('jquery'); // Ensure jQuery is loaded if needed as dependency


        // --- Use CSS ---
        $load_base_css = $params->get('social2s_load_base', true);
        $load_style_css = $params->get('s2s_load_style', 1);
        $s2s_defer_css = $params->get('social2s_css_defer', 1);
        $styleName = $this->getCalculatedStyles()->style ?? 'default'; // Get style name

        $cssOptions = ['version' => $version];
        $deferScript = '';

        // Logic for deferred CSS loading (consider simplifying or using modern techniques)
        if ($s2s_defer_css) {
            // Use a small JS loader for deferred CSS
            $buildDeferredLoader = function ($id, $url) use ($version) {
                $urlWithVersion = $url . '?v=' . $version;
                return "
                    (function() {
                        const link = document.createElement('link');
                        link.rel = 'stylesheet';
                        link.href = '{$urlWithVersion}';
                        link.id = '{$id}';
                        // Optional: Use requestAnimationFrame for slight delay
                        requestAnimationFrame(function() { document.head.appendChild(link); });
                    })();
                ";
            };

            if ($load_base_css) {
                 $baseCssUrl = $media_path . '/css/behavior/s2sdefault.css';
                 $deferScript .= $buildDeferredLoader('s2s-base-css-deferred', $baseCssUrl);
                 $this->plugin->addDebug('CSS Loading', 'Base CSS Deferred: ' . $baseCssUrl, 'info', 'css3');
            }
            if ($load_style_css) {
                 $styleCssUrl = $media_path . '/css/styles/' . $styleName . '.css';
                 $deferScript .= $buildDeferredLoader('s2s-style-css-deferred', $styleCssUrl);
                 $this->plugin->addDebug('CSS Loading', 'Style CSS Deferred: ' . $styleCssUrl, 'info', 'css3');
            }
            if ($deferScript) {
                $wa->addInlineScript($deferScript, [], ['defer' => true]);
            }
        } else {
            // Load CSS normally via WebAssetManager
            if ($load_base_css) {
                 $wa->useStyle('plg_content_social2s.base', $cssOptions);
                 $this->plugin->addDebug('CSS Loading', 'Base CSS Loaded Normally', 'info', 'css3');
            }
            if ($load_style_css) {
                 $wa->useStyle('plg_content_social2s.style.' . $styleName, $cssOptions);
                 $this->plugin->addDebug('CSS Loading', 'Style CSS Loaded Normally: ' . $styleName, 'info', 'css3');
            }
        }


        // --- Load FontAwesome (if configured) ---
        $s2s_button_style = $params->get('s2s_icon_image', 'fa');
        $s2s_button_loadFa = $params->get('social2s_load_fa6', 0);


        if ($s2s_button_style === 'fa') {
            if ($s2s_button_loadFa == 1) {
                // Load external FontAwesome 6
                $wa->useScript('plg_content_social2s.fa6.brands');
                $this->plugin->addDebug('Assets', 'External FontAwesome 6 JS Used', 'info', 'font-awesome');
            } else {
                // Use Joomla's built-in FontAwesome
                $wa->useStyle('jfontawesome');
                $this->plugin->addDebug('Assets', 'Joomla FontAwesome', 'info', 'font-awesome');
            }
        }

        // --- Set Language Info (might be needed by JS) ---
        $this->setLanguageInfo();
        // --- Set Mobile Info (might be needed by JS) ---
        $this->setMobileInfo();


        // Mark assets as loaded for this request
        $this->plugin->assetsLoaded = true;
        $this->plugin->addDebug('Assets', 'CSS/JS loading initiated (Version: ' . $version . ')', 'success', 'check-square-o');
    }

    /**
     * Sets language properties on the plugin object.
     */
    private function setLanguageInfo(): void
    {
        $app = Factory::getApplication();
        $params = $this->plugin->getParams();
        $lang = $app->getLanguage();

        $idioma_setting = $params->get('s2s_language', '0');

        if ($idioma_setting !== '0' && $idioma_setting !== '') {
            $this->plugin->langTag = $idioma_setting; // e.g., 'en-GB'
            $this->plugin->langTagShort = substr($idioma_setting, 0, 2); // e.g., 'en'
        } else {
            $this->plugin->langTag = $lang->getTag(); // e.g., 'en-GB'
            $this->plugin->langTagShort = $lang->get('tag'); // e.g., 'en' (using get('tag') often gives just 'en')
             $this->plugin->langTagShort = substr($this->plugin->langTag, 0, 2); // Fallback if get('tag') is full tag
        }

        $this->plugin->langDirection = $lang->isRTL() ? 'rtl' : 'ltr'; // Use bool result

        $this->plugin->addDebug('Language Info', 'Tag: ' . $this->plugin->langTag . ' | Short: ' . $this->plugin->langTagShort . ' | Dir: ' . $this->plugin->langDirection, 'info', 'language');
    }


    /**
     * Calculates and returns style-related settings based on context and parameters.
     *
     * @return object Object containing calculated style properties.
     */
    private function getCalculatedStyles(): object
    {
        $params = $this->plugin->getParams();
        $context = $this->plugin->getContext();

        $useArticleSettings = true; // Default to article settings

        // Check if we are in a category/featured context and if 'takefromarticle' is OFF
        if (($context === 'com_content.category' || $context === 'com_content.featured')
            && $params->get('s2s_takefromarticle', '1') === '0') {
            $useArticleSettings = false;
        }
        // Add similar checks for other contexts (VM, K2, etc.) if they have separate style settings

        // Get the appropriate parameter registry
        $styleParams = $useArticleSettings
            ? $this->plugin->getArtStyleParams()
            : $this->plugin->getCatStyleParams(); // Add logic for VM/K2 params if needed

        $styles = new \stdClass();
        $styles->sign = $styleParams->get('s2s_font_style', 'icon'); // icon, image, text
        $styles->style = $styleParams->get('s2s_style', 'default'); // default, modern, etc.
        $styles->align = $styleParams->get('s2s_button_align', 'left'); // left, right, center
        $styles->fill = $styleParams->get('s2s_fill', '0'); // 0=no, 1=yes, 2=mobile
        $styles->group = ($styleParams->get('s2s_group', '1') === '0') ? 's2s-btn-split' : 's2s-btn-group'; // 0=split, 1=group
        $styles->size = $styleParams->get('s2s_size', 's2s-btn-default'); // xs, sm, default, lg
        $styles->text = $styleParams->get('s2s_show_text', 0); // 0=no, 1=yes, 2=mobile
        $styles->balloon = $styleParams->get('s2s_balloon_pos', '0'); // 0=bottom, 1=top (or specific class names)

        // Calculate image size based on button size class
        switch ($styles->size) {
            case 's2s-btn-xs':
                $styles->image_size = '16px';
                break;
            case 's2s-btn-sm':
                $styles->image_size = '20px';
                break;
            case 's2s-btn-lg':
                $styles->image_size = '32px';
                break;
            case 's2s-btn-default':
            default:
                $styles->image_size = '24px';
                break;
        }
        $this->plugin->addDebug('Styles Calculated', 'Using ' . ($useArticleSettings ? 'Article' : 'Category/Context') . ' settings', 'info');
        return $styles;
    }


    /**
     * Generates the HTML templates for each button (icon/image + text).
     *
     * @return array<string, string> Associative array mapping social network key to its button HTML template.
     */
    private function getButtonTemplates(): array
    {
        if (!$this->styles) {
            $this->styles = $this->getCalculatedStyles(); // Ensure styles are calculated
        }

        $params = $this->plugin->getParams();
        $styles = $this->styles;

        $images_path = $this->plugin->getImagesPath(); // URL to media/plg_content_social2s/assets/
        $s2s_button_style_type = $params->get('s2s_icon_image', 'fa'); // fa, image, text (or legacy '1'?)

        $texts = $this->getTexts($styles->text);
        $buttons = [];
        $image_pixels = $styles->image_size; // e.g., '24px'


        // Define button structures based on type
        $networks = [
             'twitter'   => ['icon' => 'fab fa-x-twitter', 'text' => $texts['twitter'], 'title' => 'X (Twitter)'],
             'facebook'  => ['icon' => 'fab fa-facebook', 'text' => $texts['facebook'], 'title' => 'Facebook'],
             'pinterest' => ['icon' => 'fab fa-pinterest', 'text' => $texts['pinterest'], 'title' => 'Pinterest'],
             'linkedin'  => ['icon' => 'fab fa-linkedin', 'text' => $texts['linkedin'], 'title' => 'LinkedIn'],
             // 'gplus'     => ['icon' => 'fab fa-google-plus', 'text' => $texts['gplus'], 'title' => 'Google+'], // Deprecated
             'wapp'      => ['icon' => 'fab fa-whatsapp', 'text' => $texts['wapp'], 'title' => 'WhatsApp'],
             'tgram'     => ['icon' => 'fab fa-telegram', 'text' => $texts['tgram'], 'title' => 'Telegram'],
             'flipb'     => ['icon' => 'fab fa-flipboard', 'text' => $texts['flipb'], 'title' => 'Flipboard'],
             'vk'        => ['icon' => 'fab fa-vk', 'text' => $texts['vk'], 'title' => 'VK'],
             // 'delio'     => ['icon' => 'fab fa-delicious', 'text' => $texts['delio'], 'title' => 'Delicious'], // Often deprecated
             'snap'      => ['icon' => 'fab fa-snapchat', 'text' => $texts['snap'], 'title' => 'Snapchat'],
             'tumblr'    => ['icon' => 'fab fa-tumblr', 'text' => $texts['tumblr'], 'title' => 'Tumblr'],
             'reddit'    => ['icon' => 'fab fa-reddit', 'text' => $texts['reddit'], 'title' => 'Reddit'],
             'email'     => ['icon' => 'fas fa-envelope', 'text' => $texts['email'], 'title' => 'Email'],
        ];

        foreach ($networks as $key => $details) {
            $iconHtml = '';
            switch ($s2s_button_style_type) {
                case 'fa': // FontAwesome
                    $iconHtml = '<i class="' . ($details['icon'] ?? 'fas fa-share-alt') . '" title="' . ($details['title'] ?? '') . '"></i>';
                    break;
                case 'image': // Image (SVG assumed)
                    $imgStyle = $styles->sign ?: 'default'; // Subfolder for image style
                    $imgSrc = $images_path . 'svg/' . $imgStyle . '/' . $key . '.svg'; // Default name matching key
                     // Special case for X/Twitter
                    if ($key === 'twitter') {
                        $imgSrc = $images_path . 'svg/' . $imgStyle . '/x-twitter.svg';
                    }
                    $iconHtml = '<img loading="lazy" src="' . $imgSrc . '" alt="' . ($details['title'] ?? $key) . ' button" width="' . $image_pixels . '" height="' . $image_pixels . '"/>';
                    break;
                case 'text': // Text only (handled by $details['text'] below)
                default:
                    $iconHtml = ''; // No icon
                    break;
            }
            // Combine icon and text
            $buttons[$key] = trim($iconHtml . ' ' . $details['text']);
        }

        return $buttons;
    }


    /**
     * Generates text spans for buttons if enabled.
     *
     * @param int|string $show_text Config value (0=no, 1=yes, 2=mobile).
     * @return array<string, string> Array mapping network key to its text HTML or empty string.
     */
    private function getTexts($show_text): array
    {
        $texts = [];
        $networks = [
            'twitter' => 'SOCIAL2S_TWITTER', 'facebook' => 'SOCIAL2S_FACEBOOK', 'pinterest' => 'SOCIAL2S_PINTEREST',
            'linkedin' => 'SOCIAL2S_LINKEDIN', 'gplus' => 'SOCIAL2S_GPLUS', 'wapp' => 'SOCIAL2S_WAPP',
            'tgram' => 'SOCIAL2S_TGRAM', 'flipb' => 'SOCIAL2S_FLIPB', 'delio' => 'SOCIAL2S_DELIO',
            'snap' => 'SOCIAL2S_SNAP', 'tumblr' => 'SOCIAL2S_TUMBLR', 'reddit' => 'SOCIAL2S_REDDIT',
            'vk' => 'SOCIAL2S_VK', 'email' => 'SOCIAL2S_EMAIL',
        ];

        // Determine class based on show_text setting (applies to all texts)
        $spanClass = '';
        if ($show_text == 1) { // Always show
            $spanClass = 's2s_social_text';
        } elseif ($show_text == 2) { // Show only on mobile (JS/CSS handles visibility)
            $spanClass = 's2s_social_text s2s_text_mobile_only'; // Example class
        }
        // If show_text == 0 or invalid, $spanClass remains empty, text is effectively hidden.

        foreach ($networks as $key => $langKey) {
            $texts[$key] = $spanClass ? '<span class="' . $spanClass . '">' . Text::_($langKey) . '</span>' : '';
        }

        return $texts;
    }


    /**
     * Injects necessary data as JavaScript variables using WebAssetManager.
     */
    private function setJavascriptVariables(): void
    {
        // Check if data is prepared
        if ($this->articleData === null) {
            $this->plugin->addDebug('JS Vars', 'Skipped - Article data not prepared.', 'warning');
            return;
        }

        $params = $this->plugin->getParams();
        $wa = Factory::getDocument()->getWebAssetManager();

        // Build the data object for JS
        $jsData = [
            'debug' => (bool) $params->get('s2s_debug', 0),
            'version' => $this->_getVersion() ?: 'unknown',
            'context' => $this->plugin->getContext(),
            'isMobile' => (bool) ($this->plugin->isMobileDetected ?? false), // Use detected value
            'lang' => $this->plugin->langTag ?? 'en-GB',
            'langShort' => $this->plugin->langTagShort ?? 'en',
            'langDir' => $this->plugin->langDirection ?? 'ltr',

            // Current item data
            'pageUrl' => $this->articleData->full_link ?? '',
            'pageUrlEncoded' => $this->articleData->encoded_full_link ?? '',
            'pageTitle' => $this->articleData->title ?? '',
            'pageTitleEncoded' => $this->articleData->safetitle ?? '',
            // Pass first image URL if available and needed by JS
            'pageImage' => $this->articleData->images[0] ?? '',

            // Feature flags / settings for JS
            'checkCookie' => false, // Cookies deprecated/removed based on original code
            'loadScriptsOnLoad' => (bool) $params->get('s2s_load_scripts_onload', false),
            'ctaActive' => ($this->plugin->getContext() === 'com_content.article') ? (bool)$params->get('s2s_cta_active', 0) : false,
            'ctaDefault' => $params->get('s2s_cta_default', 'twitter'),
            'mobileBreakpoint' => (int) $params->get('s2s_art_mobile_min', 978), // For fixed position JS check
            'licenseKey' => $params->get('s2s_license_email', ''), // If needed by JS

            // Insertion parameters (if using JS-based insertion) - Consider if still needed
             'insert' => [
                 'mode' => $params->get('s2s_insert', '0'), // 0=auto, 1=manual(selector)
                 'position' => $params->get('s2s_insert_position', '0'), // 0=before, 1=after
                 'element' => $params->get('s2s_insert_element', ''), // CSS selector
             ]
        ];

        // Add the data object to the page
        $wa->addInlineScript('var s2sData = ' . json_encode($jsData) . ';');
         $this->plugin->addDebug('JS Vars', 'Data object added.', 'info', 'code');
    }


    /**
     * Gets the plugin version from the manifest cache.
     * @return string|null Plugin version string or null if not found.
     */
    private function _getVersion(): ?string
    {
        // Cache version in memory for the request
        static $version = null;
        if ($version !== null) {
            return $version === false ? null : $version; // Return cached value (or null if lookup failed before)
        }

        try {
            $db = Factory::getContainer()->get('DatabaseDriver');
            $query = $db->getQuery(true)
                ->select($db->quoteName('manifest_cache'))
                ->from($db->quoteName('#__extensions'))
                ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'))
                ->where($db->quoteName('element') . ' = ' . $db->quote('social2s'))
                ->where($db->quoteName('folder') . ' = ' . $db->quote('content'));
            $db->setQuery($query);
            $manifestCacheJson = $db->loadResult();

            if ($manifestCacheJson) {
                $manifestCache = json_decode($manifestCacheJson);
                if (isset($manifestCache->version)) {
                     $version = $manifestCache->version; // Cache success
                    return $version;
                }
            }
        } catch (\Exception $e) {
            $this->plugin->addDebug('Version Check Error', $e->getMessage(), 'warning');
        }

        $version = false; // Cache failure
        return null;
    }


    /**
     * Calculates CSS classes for the main container div.
     *
     * @return object Object containing CSS classes as a string property ('classes').
     */
    private function getSupraClasses(): object
    {
        if (!$this->styles) {
            $this->styles = $this->getCalculatedStyles(); // Ensure styles calculated
        }

        $params = $this->plugin->getParams();
        $context = $this->plugin->getContext();
        $styles = $this->styles;

        $classes = [];
        $classes[] = 's2s_wrapper'; // Base class

        // --- Add Base Behavior Class ---
        $baseBehaviorSetting = $params->get('social2s_base', '0');
        $behaviorClass = ($baseBehaviorSetting === '1') ? 's2s-behavior-modern' : 's2s-behavior-default';
        $classes[] = $behaviorClass;
        // --- End Base Behavior Class ---

        $classes[] = $this->plugin->langDirection ?? 'ltr'; // Language direction
        $classes[] = 's2s-style-' . ($styles->style ?? 'default'); // Style name
        $classes[] = 's2s-icons-' . ($params->get('s2s_icon_image', 'fa') ?? 'fa'); // Icon type
        $classes[] = $styles->size ?? 's2s-btn-default'; // Size class
        $classes[] = $styles->group ?? 's2s-btn-group'; // Group/split class
        $classes[] = 's2s-align-' . ($styles->align ?? 'left'); // Alignment

        // Text visibility class
        if ($styles->text == 0) {
            $classes[] = 's2s_no_text';
        } elseif ($styles->text == 1) {
            $classes[] = 's2s_text_always';
        } else { // Assumes 2 = mobile only
            $classes[] = 's2s_text_mobile';
        }

        // Fill class
        if ($styles->fill == 1) {
            $classes[] = 's2s_fill_always';
        } elseif ($styles->fill == 2) {
            $classes[] = 's2s_fill_mobile';
        }

        // Balloon position
        $classes[] = ($styles->balloon == '1') ? 's2s_balloon_top' : 's2s_balloon_bottom';

        // Fixed position class (apply only in article context if configured)
        $s2s_pos_fix_setting = $params->get('s2s_art_fixed', 0);
        if ($s2s_pos_fix_setting && $context === 'com_content.article') {
            $mode = $params->get('s2s_art_fixed_mode', 's2s_fixed_horizontal');
            $posx = $params->get('s2s_art_fixed_posx', 'right');
            $posy = $params->get('s2s_art_fixed_posy', 'center');
            $classes[] = 's2s_pos_fixed';
            $classes[] = $mode; // horizontal or vertical
            $classes[] = 's2s_pos_fix_x_' . $posx;
            $classes[] = 's2s_pos_fix_y_' . $posy;
            // Adjust button group class if vertical fixed
            if ($mode === 's2s_fixed_vertical') {
                // Find and replace group class if needed
                $groupClassKey = array_search('s2s-btn-group', $classes);
                if ($groupClassKey !== false) {
                    $classes[$groupClassKey] = 's2s-btn-group-vertical';
                } else {
                    $splitClassKey = array_search('s2s-btn-split', $classes);
                    // Use appropriate vertical class name if needed
                    if ($splitClassKey !== false) {
                        $classes[$splitClassKey] = 's2s-btn-split-vertical';
                    }
                }
            }
        }

        // Sticky class (example logic, adjust based on actual parameters)
        $stickySetting = $params->get('s2s_art_sticky', '0'); // Assuming context check similar to 'fill'
        if ($context !== 'com_content.article' && !$params->get('s2s_takefromarticle', '1')) {
             $stickySetting = $params->get('s2s_cat_sticky', '0'); // Example for category
        }
        if ($stickySetting == 1) {
            $classes[] = 's2s_sticky_always';
        } elseif ($stickySetting == 2) {
            $classes[] = 's2s_sticky_mobile';
        }

        // Cookies deprecated
        // $checkCookie = $params->get('s2s_stupid_cookie_on', '0');
        // if ($checkCookie) { $classes[] = 'cookie_on'; }

        // Filter out empty values and ensure uniqueness
        $finalClasses = array_unique(array_filter($classes));

        $this->plugin->addDebug('Supra Classes', implode(' ', $finalClasses), 'info', 'css3');

        // Return as object with 'classes' string property
        return (object) ['classes' => implode(' ', $finalClasses)];
    }



    /**
     * Generates the final HTML for the ordered social buttons.
     *
     * @return string HTML output for the buttons.
     */
    private function generateButtonsHtml(): string
    {
        // Check if data is prepared
        if ($this->articleData === null) {
            $this->plugin->addDebug('Generate Buttons', 'Skipped - Article data not prepared.', 'warning');
            return '';
        }
        // Ensure button templates are ready
        if (empty($this->buttonTemplates)) {
            $this->buttonTemplates = $this->getButtonTemplates();
        }


        $params = $this->plugin->getParams();
        $orderSetting = $params->get('s2s_order', '0,1,2,3,4,5,6,7,8,9,10,11,12,13');
        $s2s_order = array_map('trim', explode(',', $orderSetting));

        // Map order numbers to network keys
        $orderMap = [
            '0' => 'twitter', '1' => 'facebook', '2' => 'pinterest', '3' => 'linkedin',
            '4' => null, // gplus removed
            '5' => 'wapp', '6' => 'tgram', '7' => 'flipb', '8' => 'snap',
            '9' => 'tumblr', '10' => 'vk', '11' => 'email', '12' => 'reddit',
            '13' => 'more' // Placeholder for the 'more' button logic
        ];

        $networkParams = [
            'twitter' => 'twitter_on', 'facebook' => 'facebook_on', 'pinterest' => 'pinterest_on',
            'linkedin' => 'linkedin_on', 'wapp' => 'wapp_on', 'tgram' => 'tgram_on',
            'flipb' => 'flipb_on', 'snap' => 'snap_on', 'tumblr' => 'tumblr_on',
            'vk' => 'vk_on', 'email' => 'email_on', 'reddit' => 'reddit_on',
            'more' => 'mas_on'
        ];

        $html = '';
        $s2sSocialHelper = new S2sSocial($this->plugin, $this->articleData, $this->buttonTemplates, $this->styles); // Pass data

        $isMoreButtonActive = false;
        $param_mas_on = $params->get('mas_on', 0);
        if ($param_mas_on == 1) { // Always on
            $isMoreButtonActive = true;
        } elseif ($param_mas_on == 2 && ($this->plugin->isMobileDetected ?? false)) { // Mobile only
            $isMoreButtonActive = true;
        }

        $buttonsInMore = [];

        foreach ($s2s_order as $orderKey) {
            $networkKey = $orderMap[$orderKey] ?? null;
            if ($networkKey === null) {
                continue; // Skip invalid keys or removed networks like G+
            }

            $paramName = $networkParams[$networkKey] ?? null;
            if ($paramName === null) {
                continue; // Skip if no parameter defined
            }

            $isEnabled = (bool) $params->get($paramName, ($networkKey === 'more' ? 0 : 1)); // Default most to ON, 'more' to OFF

            if ($isEnabled) {
                $methodName = 'get' . ucfirst($networkKey); // e.g., getTwitter

                 // Check if this button should be hidden behind 'more'
                 $isBehindMore = ($isMoreButtonActive && $networkKey !== 'more' && $orderKey !== end($s2s_order));


                if (method_exists($s2sSocialHelper, $methodName)) {
                    if ($isBehindMore) {
                        // Generate button but store it for the 'more' popup/dropdown
                        $buttonsInMore[$networkKey] = $s2sSocialHelper->$methodName('s2s-in-more'); // Pass class/flag
                    } else {
                        // Generate button directly
                        $html .= $s2sSocialHelper->$methodName();
                    }
                } elseif ($networkKey === 'more' && $isMoreButtonActive) {
                    // Generate the actual 'more' button/trigger itself
                    // It needs the $buttonsInMore content
                    $html .= $s2sSocialHelper->getMore($buttonsInMore);
                    // Once the 'more' button is added, stop processing further items as 'behind more'
                    $isMoreButtonActive = false; // Turn off the flag after adding the button
                }
            }
        }

        return $html;
    }
}
// End of S2sTmpl class
